package io.gitee.declear.dec.cloud.common.rpc.protocol.code;

import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.exception.DecCloudServiceException;
import io.gitee.declear.dec.cloud.common.remoting.resource.DecCloudApi;
import io.gitee.declear.dec.cloud.common.remoting.DecRemoteContext;
import io.gitee.declear.dec.cloud.common.rpc.protocol.pack.DecCloudProtocolPack;
import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.ChannelOutboundHandlerAdapter;
import io.netty.handler.codec.ByteToMessageDecoder;
import io.netty.handler.codec.MessageToByteEncoder;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * netty 编码/解码器: dec-cloud协议编解码器
 * @author DEC
 */
@Slf4j
public class DecCloudCodeAdapter {

    private static final Integer MIN_SIZE_OF_BYTE_BUF = 4;

    private final ChannelOutboundHandlerAdapter decCloudEncoder = new DecCloudCodeAdapter.InternalEncoder();

    private final ChannelInboundHandlerAdapter decCloudDecoder = new DecCloudCodeAdapter.InternalDecoder();

    public ChannelOutboundHandlerAdapter getEncoder() {
        return decCloudEncoder;
    }

    public ChannelInboundHandlerAdapter getDecoder() {
        return decCloudDecoder;
    }

    private class InternalEncoder extends MessageToByteEncoder<DecCloudProtocolPack> {

        @Override
        protected void encode(ChannelHandlerContext ctx, DecCloudProtocolPack pack, ByteBuf out) throws Exception {
            out.writeShort(pack.getType());
            out.writeBytes(pack.getVersion().getBytes(StandardCharsets.UTF_8));
            out.writeBytes(pack.getId().getBytes(StandardCharsets.UTF_8));

            // write cloudOrigin
            if(pack.getCloudOrigin() != null) {
                out.writeShort(1);
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                ObjectOutputStream oos = new ObjectOutputStream(bos);
                oos.writeObject(pack.getCloudOrigin());
                byte[] originBytes = bos.toByteArray();
                out.writeInt(originBytes.length);
                out.writeBytes(originBytes);
            } else {
                out.writeShort(0);
            }
            // write cloudDestination
            if(pack.getCloudDestination() != null) {
                out.writeShort(1);
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                ObjectOutputStream oos = new ObjectOutputStream(bos);
                oos.writeObject(pack.getCloudDestination());
                byte[] destinationBytes = bos.toByteArray();
                out.writeInt(destinationBytes.length);
                out.writeBytes(destinationBytes);
            } else {
                out.writeShort(0);
            }

            if(CommonUtils.isNotEmpty(pack.getDataList())) {
                out.writeInt(pack.getDataList().size());
                byte[][] dataBytes = new byte[pack.getDataList().size()][];
                for (int i = 0; i < pack.getDataList().size(); i++) {
                    ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    ObjectOutputStream oos = new ObjectOutputStream(bos);
                    oos.writeObject(pack.getDataList().get(i));
                    dataBytes[i] = bos.toByteArray();
                }
                for (byte[] dataByte : dataBytes) {
                    out.writeInt(dataByte.length);
                }
                for (byte[] dataByte : dataBytes) {
                    out.writeBytes(dataByte);
                }
            } else {
                out.writeInt(0);
            }
        }
    }

    private class InternalDecoder extends ByteToMessageDecoder {

        @Override
        protected void decode(ChannelHandlerContext ctx, ByteBuf input, List<Object> out) throws Exception {
            if(input.readableBytes() < MIN_SIZE_OF_BYTE_BUF) {
                ctx.fireChannelRead(input.retain());
                return;
            }
            String protocol = readString(input, MIN_SIZE_OF_BYTE_BUF);
            if(!Objects.equals(protocol, Constants.DEC_CLOUD_CODE_PROTOCOL_NAME)) {
                input.readerIndex(input.readerIndex() - MIN_SIZE_OF_BYTE_BUF);
                ctx.fireChannelRead(input.retain());
                return;
            }

            short type = input.readShort();
            String version = readString(input, Constants.DEC_CLOUD_CODE_PROTOCOL_VERSION_LENGTH);
            String id = readString(input, Constants.DEC_CLOUD_CODE_PROTOCOL_PACK_ID_LENGTH);
            DecCloudApi cloudOrigin = null, cloudDestination = null;
            if(input.readShort() == 1) {
                cloudOrigin = processCloudApi(input);
            }
            if(input.readShort() == 1) {
                cloudDestination = processCloudApi(input);
            }
            List<Serializable> dataList = processDataList(input);

            DecRemoteContext<Serializable> context = new DecRemoteContext<>();
            context.setId(id);
            context.setType(type);
            if(Objects.equals(Constants.DEC_CLOUD_CODE_PROTOCOL_VERSION, version)) {
                switch (type) {
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_1:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_4:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_6:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_8:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_10:
                        context.setCloudOrigin(cloudOrigin);
                        context.setCloudDestination(cloudDestination);
                        context.setParamList(dataList);
                        out.add(context);
                        break;
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_2:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_5:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_7:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_9:
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_11:
                        context.setBackStatus((Short) dataList.get(0));
                        if(Objects.equals(context.getBackStatus(), DecRemoteContext.REMOTE_CONTEXT_BACK_STATUS_SUCCESS)) {
                            context.setResult(dataList.get(1));
                        } else {
                            context.setFailure((Throwable) dataList.get(1));
                        }
                        out.add(context);
                        break;
                    case Constants.DEC_CLOUD_CODE_PROTOCOL_TYPE_3:
                        break;
                    default:
                        break;
                }
            } else {
                log.error("dec cloud protocol version not match.");
                throw new DecCloudServiceException("dec cloud protocol version not match.");
            }
        }

        private List<Serializable> processDataList(ByteBuf input) throws IOException, ClassNotFoundException {
            int dataNum = input.readInt();
            List<Serializable> dataList = null;
            if(dataNum > 0) {
                dataList = new ArrayList<>(dataNum);
                int[] dataLengthArray = new int[dataNum];
                for (int i = 0; i < dataNum; i++) {
                    dataLengthArray[i] = input.readInt();
                }
                for (int i = 0; i < dataNum; i++) {
                    byte[] objectBytes = new byte[dataLengthArray[i]];
                    input.readBytes(objectBytes);
                    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(objectBytes));
                    Object object = ois.readObject();

                    dataList.add((Serializable) object);
                }
            }

            return dataList;
        }

        private String readString(ByteBuf input, int length) {
            byte[] bytes = new byte[length];
            input.readBytes(bytes);
            return new String(bytes, StandardCharsets.UTF_8);
        }

        private DecCloudApi processCloudApi(ByteBuf input) throws IOException, ClassNotFoundException {
            int length = input.readInt();

            byte[] objectBytes = new byte[length];
            input.readBytes(objectBytes);
            ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(objectBytes));
            Object object = ois.readObject();

            return (DecCloudApi) object;
        }
    }

}
