package io.gitee.declear.dec.cloud.common.rpc.protocol.code;

import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.ChannelOutboundHandlerAdapter;
import io.netty.handler.codec.LengthFieldBasedFrameDecoder;
import io.netty.handler.codec.LengthFieldPrepender;
import lombok.extern.slf4j.Slf4j;

import java.nio.charset.StandardCharsets;
import java.util.List;

/**
 * netty 编码/解码器: dec-cloud/http/https 协议数据包分发器
 * @author DEC
 */
@Slf4j
public class DecCloudPackageCodeAdapter {

    private final ChannelOutboundHandlerAdapter packEncoder = new DecCloudPackageCodeAdapter.InternalEncoder(4);

    private final ChannelInboundHandlerAdapter packDecoder = new DecCloudPackageCodeAdapter.InternalDecoder(512 * 1024, 0, 4, 0, 4);

    public ChannelOutboundHandlerAdapter getEncoder() {
        return packEncoder;
    }

    public ChannelInboundHandlerAdapter getDecoder() {
        return packDecoder;
    }

    private class InternalEncoder extends LengthFieldPrepender {

        public InternalEncoder(int lengthFieldLength) {
            super(lengthFieldLength);
        }

        @Override
        protected void encode(ChannelHandlerContext ctx, ByteBuf msg, List<Object> out) throws Exception {
            out.add(ctx.alloc().buffer(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH).writeBytes(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME.getBytes(StandardCharsets.UTF_8)));
            super.encode(ctx, msg, out);
        }
    }

    private class InternalDecoder extends LengthFieldBasedFrameDecoder {

        public InternalDecoder(int maxFrameLength, int lengthFieldOffset, int lengthFieldLength, int lengthAdjustment, int initialBytesToStrip) {
            super(maxFrameLength, lengthFieldOffset, lengthFieldLength, lengthAdjustment, initialBytesToStrip);
        }

        @Override
        protected Object decode(ChannelHandlerContext ctx, ByteBuf in) throws Exception {
            if(in.readableBytes() < Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH) {
                return in.retain();
            }

            byte[] bytes = new byte[Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH];
            in.getBytes(in.readerIndex(), bytes);
            String protocol = new String(bytes, StandardCharsets.UTF_8);

            if(protocol.startsWith(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME)) {
                in.skipBytes(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH);
                ByteBuf decProtocolBuf = (ByteBuf) super.decode(ctx, in);
                if(null != decProtocolBuf) {
                    return ctx.alloc().buffer(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH + decProtocolBuf.readableBytes()).writeBytes(Constants.DEC_CLOUD_CODE_PROTOCOL_NAME.getBytes(StandardCharsets.UTF_8)).writeBytes(decProtocolBuf);
                } else  {
                    in.readerIndex(in.readerIndex() - Constants.DEC_CLOUD_CODE_PROTOCOL_NAME_LENGTH);
                }

                return null;
            } else {
                ByteBuf frame = in.retainedDuplicate();
                in.skipBytes(in.readableBytes());
                return frame;
            }
        }
    }

}
