package io.gitee.declear.dec.cloud.common.web;

import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http.FullHttpRequest;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 创建 DecHttpRequest/HttpResponse 的工具类
 * @author DEC
 */
public class DecHttpAttributeBuilder {

    private static final Map<String, DecHttpSession> HTTP_SESSION_MAP = new ConcurrentHashMap<>(2 ^ 10);

    public static DecHttpRequest buildHttpRequest(String uri) {
        return buildHttpRequest(uri, DecHttpMethod.GET);
    }

    public static DecHttpRequest buildHttpRequest(String uri, DecHttpMethod method) {
        return new DecHttpRequest(uri, method, DecHttpVersion.HTTP_1_1, DecHttpHeader.of(new HashMap<>(2 ^ 4)));
    }

    public static DecHttpRequest buildHttpRequest(FullHttpRequest fullHttpRequest) {
        DecHttpRequest request = new DecHttpRequest(fullHttpRequest.uri(), DecHttpMethod.valueOf(fullHttpRequest.method().name()),
                DecHttpVersion.ofVersion(fullHttpRequest.protocolVersion().toString()), DecHttpHeader.of(fullHttpRequest.headers()));

        ByteBuf buf = fullHttpRequest.content();
        byte[] bytes = new byte[buf.readableBytes()];
        buf.readBytes(bytes);
        request.setContent(new String(bytes, StandardCharsets.UTF_8));

        return request;
    }

    public static DecHttpResponse buildHttpResponse(DecHttpRequest httpRequest, DecHttpResponseStatus status) {
        return new DecHttpResponse(httpRequest.version(), status);
    }

    public static DecHttpSession searchSession(DecHttpRequest httpRequest) {
        if(null != httpRequest.sessionId() && null != HTTP_SESSION_MAP.get(httpRequest.sessionId())) {
            DecHttpSession session = HTTP_SESSION_MAP.get(httpRequest.sessionId());
            if(Duration.between(LocalDateTime.now(), session.getLastAccessTime()).toMinutes() < Constants.DEC_CLOUD_WEB_SERVER_SESSION_EXPIRE_TIME_MINUTES) {
                return session;
            }
        }
        DecHttpSession session = new DecHttpSession();
        HTTP_SESSION_MAP.put(session.getSessionId(), session);
        return session;
    }

}
