package io.gitee.declear.dec.cloud.common.web;

import lombok.Data;

/**
 * dec cloud 框架中的 HttpCookie
 * @author DEC
 */
@Data
public class DecHttpCookie {

    public DecHttpCookie(String name, String value) {
        this.name = name;
        this.value = value;
    }

    /**
     * Name for the cookie.
     */
    private String name;

    /**
     * Value for the cookie.
     */
    private String value;

    /**
     * Domain for the cookie.
     */
    private String domain;

    /**
     * Path of the cookie.
     */
    private String path = "/";

    /**
     * Whether to use "HttpOnly" cookies for the cookie.
     */
    private Boolean httpOnly = true;

    /**
     * Whether to always mark the cookie as secure.
     */
    private Boolean secure = false;

    /**
     * expires time
     */
    private String expires;

    /**
     * Maximum age of the cookie. If a duration suffix is not specified, seconds will be
     * used. A positive value indicates when the cookie expires relative to the current
     * time. A value of 0 means the cookie should expire immediately. A negative value
     * means no "Max-Age".
     */
    private Long maxAge;

    /**
     * SameSite setting for the cookie.
     */
    private SameSite sameSite = SameSite.STRICT;

    private Priority priority = Priority.LOW;

    public String cookieString() {
        StringBuilder builder = new StringBuilder(name + '=' + value + ';');
        if (null != domain) {
            builder.append(" Domain=" + domain + ';');
        }
        builder.append(" Path=" + path + ';');
        if (httpOnly.equals(true)) {
            builder.append(" HttpOnly;");
        }
        if (secure.equals(true)) {
            builder.append(" Secure=" + secure + ';');
        }
        if(null != expires) {
            builder.append(" Expires=" + expires + ';');
        }
        if(null != maxAge) {
            builder.append(" Max-Age=" + maxAge + ';');
        }
        builder.append(" SameSite=" + sameSite.value + ';');
        builder.append(" Priority=" + priority.value + ';');
        return builder.toString();
    }

    public enum SameSite {

        /**
         * Cookies are sent in both first-party and cross-origin requests.
         */
        NONE("None"),

        /**
         * Cookies are sent in a first-party context, also when following a link to the
         * origin site.
         */
        LAX("Lax"),

        /**
         * Cookies are only sent in a first-party context (i.e. not when following a link
         * to the origin site).
         */
        STRICT("Strict");

        private final String value;

        SameSite(String value) {
            this.value = value;
        }

        public String value() {
            return this.value;
        }

    }

    public enum Priority {

        /**
         * Cookies priority: Low.
         */
        LOW("Low"),

        /**
         * Cookies priority: Medium.
         */
        MEDIUM("Medium"),

        /**
         * Cookies priority: High.
         */
        HIGH("High");

        private final String value;

        Priority(String value) {
            this.value = value;
        }

        public String value() {
            return this.value;
        }

    }

}
