package io.gitee.declear.dec.cloud.common.web.context;

import com.alibaba.fastjson2.JSONObject;
import io.gitee.declear.common.utils.CommonUtils;
import io.gitee.declear.dec.cloud.common.constants.Constants;
import io.gitee.declear.dec.cloud.common.exception.DecCloudServiceException;
import io.gitee.declear.dec.cloud.common.mono.DecCloudContextMono;
import io.gitee.declear.dec.cloud.common.property.PropertiesManager;
import io.gitee.declear.dec.cloud.common.web.*;
import io.gitee.declear.dec.cloud.common.web.bind.annotation.RequestBody;
import io.gitee.declear.dec.cloud.common.web.bind.annotation.RequestParam;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import lombok.Data;
import org.springframework.context.ApplicationContext;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.time.LocalDateTime;
import java.util.Objects;

/**
 * 一条http请求的上下文； 包括 HttpRequest/HttpResponse/HttpSession
 * @author DEC
 */
@Data
public class DecWebContext {

    private String id;

    private DecHttpRequest httpRequest;

    private DecHttpResponse httpResponse;

    private DecHttpSession httpSession;

    private ChannelHandlerContext channelHandlerContext;

    private LocalDateTime receiveTime;

    public DecWebContext() {
        this.id = CommonUtils.UUID();
    }

    public void doService(DecWebContextManager decWebContextManager, ApplicationContext applicationContext){
        // web context holder thread local set value
        DecWebContextHolder.setWebContext(this);

        try {
            Method method = decWebContextManager.getUriServiceMethod(httpRequest);
            if(null == method) {
                httpResponse = DecHttpAttributeBuilder.buildHttpResponse(httpRequest, DecHttpResponseStatus.NOT_FOUND);
                httpResponse.setContent(JSONObject.toJSONString(DecHttpResponseStatus.NOT_FOUND.toString()));
                channelHandlerContext.writeAndFlush(httpResponse).addListener(ChannelFutureListener.CLOSE);
                return;
            }

            Object result = processResult(applicationContext, method);

            httpResponse = DecHttpAttributeBuilder.buildHttpResponse(httpRequest, DecHttpResponseStatus.OK);
            if(!Objects.equals(httpRequest.sessionId(), httpSession.getSessionId())) {
                PropertiesManager propertiesManager = applicationContext.getBean(PropertiesManager.class);
                DecHttpCookie sessionIdCookie = new DecHttpCookie(Constants.DEC_CLOUD_WEB_SERVER_SESSION_ID, httpSession.getSessionId());
                String contextPath = propertiesManager.getProperty(Constants.DEC_CLOUD_CONTEXT_PATH);
                if(CommonUtils.isNotEmpty(contextPath)) {
                    if (contextPath.charAt(0) != Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        contextPath = Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER + contextPath;
                    }
                    if (contextPath.charAt(contextPath.length() - 1) == Constants.DEC_CLOUD_HTTP_URI_SPLIT_CHARACTER) {
                        sessionIdCookie.setPath(contextPath.substring(0, contextPath.length() - 1));
                    } else {
                        sessionIdCookie.setPath(contextPath);
                    }
                }

                httpResponse.header().set(Constants.DEC_CLOUD_WEB_SERVER_SET_COOKIE, sessionIdCookie.cookieString());
            }
            if(CommonUtils.isNotEmpty(httpResponse.cookies())) {
                httpResponse.cookies().forEach(cookie -> httpResponse.header().set(Constants.DEC_CLOUD_WEB_SERVER_SET_COOKIE, cookie.cookieString()));
            }
            if(result instanceof DecCloudContextMono<?>) {
                ((DecCloudContextMono<?>) result).onComplete(obj -> {
                    httpResponse.setContent(JSONObject.toJSONString(obj));
                    channelHandlerContext.writeAndFlush(httpResponse).addListener(ChannelFutureListener.CLOSE);
                });
            } else {
                httpResponse.setContent(JSONObject.toJSONString(result));
                channelHandlerContext.writeAndFlush(httpResponse).addListener(ChannelFutureListener.CLOSE);
            }
        } catch (Exception e) {
            httpResponse = DecHttpAttributeBuilder.buildHttpResponse(httpRequest, DecHttpResponseStatus.INTERNAL_SERVER_ERROR);
            channelHandlerContext.writeAndFlush(httpResponse).addListener(ChannelFutureListener.CLOSE);
            throw new DecCloudServiceException(e);
        }
    }

    private Object processResult(ApplicationContext applicationContext, Method method) throws InvocationTargetException, IllegalAccessException {
        Object result;
        Class<?>[] types = method.getParameterTypes();
        Annotation[][] parameterAnnotations = method.getParameterAnnotations();
        Object ctl = applicationContext.getBean(method.getDeclaringClass());
        if(types.length > 0) {
            int requestBodyNum = 0;
            Object[] args = new Object[types.length];
            for (int i = 0; i < types.length; i++) {
                if(Objects.equals(types[i].getTypeName(), DecHttpRequest.class.getTypeName())) {
                    args[i] = httpRequest;
                } else if(Objects.equals(types[i].getTypeName(), DecHttpResponse.class.getTypeName())) {
                    args[i] = httpResponse;
                } else if(Objects.equals(types[i].getTypeName(), DecHttpSession.class.getTypeName())) {
                    args[i] = httpSession;
                } else {
                    RequestParam requestParam = queryAnnotation(parameterAnnotations[i], RequestParam.class);
                    if(requestParam != null) {
                        args[i] = httpRequest.params().get(requestParam.name());
                    } else {
                        RequestBody requestBody = queryAnnotation(parameterAnnotations[i], RequestBody.class);
                        if (requestBody != null) {
                            if (requestBodyNum > 0) {
                                throw new IllegalArgumentException(String.format("%s#%s: RequestBody should only be one.", method.getDeclaringClass(), method.getName()));
                            }
                            requestBodyNum++;
                            args[i] = JSONObject.parseObject(httpRequest.content(), types[i]);
                        } else {
                            if (types.length == 1 && CommonUtils.isNotEmpty(httpRequest.content())) {
                                args[i] = JSONObject.parseObject(httpRequest.content(), types[i]);
                            }
                        }
                    }
                }
            }
            result = method.invoke(ctl, args);
        } else {
            result = method.invoke(ctl);
        }
        return result;
    }

    private <T> T queryAnnotation(Annotation[] annotations, Class<T> type) {
        for (Annotation annotation : annotations) {
            if(Objects.equals(annotation.annotationType().getTypeName(), type.getTypeName())) {
                return (T) annotation;
            }
        }
        return null;
    }
}
