package io.gitee.declear.dec.cloud.common.web.context;

import io.gitee.declear.common.utils.NamedThreadFactory;
import io.gitee.declear.dec.cloud.common.property.HardWareInfoManager;
import org.springframework.context.ApplicationContext;

import java.util.concurrent.*;

/**
 * DecWebContext Processor
 * @author DEC
 */
public class DecWebContextProcessor {

    private ApplicationContext applicationContext;

    private HardWareInfoManager hardWareInfoManager;

    private DecWebContextManager webContextManager;

    /**
     * 处理 DecWebContext 的线程池
     */
    private ThreadPoolExecutor threadPool;

    /**
     * 管理线程池的核心线程数的定时任务
     */
    private ScheduledExecutorService coreSizeSchedule;

    /**
     * 管理线程池中正在执行任务的定时任务，终止出问题的web请求（牧羊犬定时任务）
     */
    private ScheduledExecutorService shepherdSchedule;

    /**
     * 线程池初始大小
     */
    private Integer initThreadPoolCoreSize = 200;

    private static final Integer THREAD_POOL_CORE_SIZE_INCREMENT = 5;
    private static final Integer THREAD_POOL_CORE_SIZE_INCREMENT_TWOFOLD = 10;
    private static final Float CPU_AND_MEMORY_BUSY_THRESHOLD = 90.0f;

    public DecWebContextProcessor(ApplicationContext applicationContext, HardWareInfoManager hardWareInfoManager, DecWebContextManager webContextManager) {
        this.applicationContext = applicationContext;
        this.hardWareInfoManager = hardWareInfoManager;
        this.webContextManager = webContextManager;

        ThreadFactory namedThreadFactory = new NamedThreadFactory("w-context-pool-");
        threadPool = new ThreadPoolExecutor(initThreadPoolCoreSize, Integer.MAX_VALUE, 120L, TimeUnit.SECONDS,
                new SynchronousQueue<>(), namedThreadFactory);

        // schedule-pool-cp
        coreSizeSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-cp-w"));
        coreSizeSchedule.scheduleWithFixedDelay(() -> {
            manageThreadPoolSize();
        }, 10,5, TimeUnit.SECONDS);

        // schedule-pool-cp-s
        shepherdSchedule = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("schedule-pool-cp-s"));
        coreSizeSchedule.scheduleWithFixedDelay(() -> {
            manageBadWebRequest();
        }, 10,5, TimeUnit.SECONDS);
    }

    public void shutdown() {
        if(threadPool != null) {
            threadPool.shutdown();
        }
        if(null != coreSizeSchedule) {
            coreSizeSchedule.shutdown();
        }
        if(null != shepherdSchedule) {
            shepherdSchedule.shutdown();
        }
    }

    private void manageThreadPoolSize() {
        if(threadPool.getQueue().size() > 0 && hardWareInfoManager.getCpuOccupyRate() < CPU_AND_MEMORY_BUSY_THRESHOLD && hardWareInfoManager.getMemoryOccupyRate() < CPU_AND_MEMORY_BUSY_THRESHOLD) {
            // 线程池排队多时，增加核心线程
            threadPool.setCorePoolSize(threadPool.getCorePoolSize() + THREAD_POOL_CORE_SIZE_INCREMENT);
        } else if(threadPool.getCorePoolSize() > initThreadPoolCoreSize && threadPool.getQueue().size() == 0 && threadPool.getActiveCount() < threadPool.getCorePoolSize() - THREAD_POOL_CORE_SIZE_INCREMENT_TWOFOLD) {
            // 线程池空闲时，减少核心线程，但始终不少于初始线程
            threadPool.setCorePoolSize(threadPool.getCorePoolSize() - THREAD_POOL_CORE_SIZE_INCREMENT);
        }
    }

    /**
     * 管理线程池中正在执行任务的定时任务
     */
    private void manageBadWebRequest() {

    }

    public void processDecWebContext(DecWebContext context) {
        threadPool.submit(() -> {
            context.doService(webContextManager, applicationContext);
            webContextManager.removeWebContext(context.getId());
        });
    }

}
