package com.dss.sdk.api.client;

import com.dss.sdk.api.base.BaseApi;
import com.dss.sdk.api.req.*;
import com.dss.sdk.api.resp.*;
import com.dss.sdk.api.service.DssClientService;
import com.dss.sdk.constants.ApiUrlConstants;
import com.dss.sdk.exception.ApiException;
import com.dss.sdk.response.DssResponse;
import lombok.extern.slf4j.Slf4j;

/**
 * @author Fadada
 */
public class FileSignClient extends BaseApi {

    public FileSignClient(DssClientService clientService) {
        super(clientService);
    }

    /**
     * POST /fileSign/sign : 文件签署
     * 接口说明：对文件进行电子签名，支持把需要加盖的电子签名及证书，加盖到指定的签署位置。支持电子印章、日期章、骑缝章及签署备注。 适用场景：对最终确认签署的电子文件（PDF/OFD）进行电子签名。
     *
     * @param fileSignSignRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<FileSignSignResponse> fileSignSign(FileSignSignRequest fileSignSignRequest) throws ApiException {
        return clientService.execute(fileSignSignRequest, ApiUrlConstants.fileSignSign, FileSignSignResponse.class);
    }

    /**
     * POST /api/file/sign/across : 骑缝章签署
     *
     * @param fileSignSignRequest
     * @return
     */
    public DssResponse<FileSignSignResponse> fileSignAcross(FileSignSignRequest fileSignSignRequest) throws ApiException {
        return clientService.execute(fileSignSignRequest, ApiUrlConstants.fileSignAcross, FileSignSignResponse.class);
    }

    /**
     * POST /api/file/sign/remark : 备注签署
     *
     * @param fileSignSignRequest
     * @return
     */
    public DssResponse<FileSignSignResponse> fileSignRemark(FileSignSignRequest fileSignSignRequest) throws ApiException {
        return clientService.execute(fileSignSignRequest, ApiUrlConstants.fileSignRemark, FileSignSignResponse.class);
    }

    /**
     * POST /api/sign/verify/sms/send : 获取签署短信验证码
     * 接口说明
     * 接口说明：调用该接口可以向指定手机号发送短信验证码
     * 适用场景：1、需要验证码做意愿校验的场景，获取验证码后，使用相同的transactionId调用[文件签署]接口，并回填短信验证码。
     *
     * 特别说明：1、允许相同transactionId多次调用获取验证码接口，每次调用返回不同的验证码，仅最新的验证码有效。
     * 2、同一个transactionId获取验证码后，文件签署时必须回填验证码，否则不能完成签署
     * @param sendSmsRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<Boolean> sendSms(SendSmsRequest sendSmsRequest) throws ApiException {
        return clientService.execute(sendSmsRequest, ApiUrlConstants.signSendSms, Boolean.class);
    }

    /**
     * POST /api/sign/verify/face/url : 文件签署（刷新获取刷脸或互动视频链接）
     *
     * @param signVerifyUrlRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<UrlResponse> signVerifyUrl(SignVerifyUrlRequest signVerifyUrlRequest) throws ApiException {
        return clientService.execute(signVerifyUrlRequest, ApiUrlConstants.signVerifyUrl, UrlResponse.class);
    }

    /**
     * POST /api/sign/verify/face/url : 文件签署（刷脸校验）
     * 接口说明：对文件进行电子签名，支持把需要加盖的电子签名及证书，加盖到指定的签署位置。支持电子印章、日期章、骑缝章及签署备注。
     * 适用场景：1、适用于使用刷脸做意愿校验签署文件场景。
     * 注意事项:
     * 1、确保签署的印章内容和证书主体的信息保持一致，避免纠纷。
     * 2、平台方确认用户签署时的真实意愿后，再调用此接口完成签章。
     *
     * @param faceSignRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<FaceSignResponse> faceSign(FaceSignRequest faceSignRequest) throws ApiException {
        return clientService.execute(faceSignRequest, ApiUrlConstants.faceSign, FaceSignResponse.class);
    }

    /**
     * POST /api/sign/verify/video/url : 文件签署（刷脸校验）
     * 接口说明：对文件进行电子签名，支持把需要加盖的电子签名及证书，加盖到指定的签署位置。支持电子印章、日期章、骑缝章及签署备注。
     * 适用场景：1、适用于使用刷脸做意愿校验签署文件场景。
     * 注意事项:
     * 1、确保签署的印章内容和证书主体的信息保持一致，避免纠纷。
     * 2、平台方确认用户签署时的真实意愿后，再调用此接口完成签章。
     *
     * @param videoSignRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<VideoSignResponse> videoSign(VideoSignRequest videoSignRequest) throws ApiException {
        return clientService.execute(videoSignRequest, ApiUrlConstants.videoSign, VideoSignResponse.class);
    }

    /**
     * 文件批量签署-支持互动视频校验
     *
     * @param request 批量签名请求对象
     * @return DssResponse<VideoSignResponse> 互动视频验证地址信息
     * @throws ApiException 如果请求失败，抛出异常
     */
    public DssResponse<VideoSignResponse> videoBatchSign(VideoBatchSignRequest request) throws ApiException {
        return clientService.execute(request, ApiUrlConstants.videoBatchSign, VideoSignResponse.class);
    }

    /**
     * POST /api/sign/verify/result : 查询意愿校验结果
     * 接口说明：
     * 使用[文件签署-支持刷脸校验][文件签署-支持互动视频校验]接口时，主动查询意愿校验结果
     *
     * @param verifyResultRequest (optional)
     * @return 成功 (status code 200)
     */
    public DssResponse<VerifyResultResponse> verifyResult(VerifyResultRequest verifyResultRequest) throws ApiException {
        return clientService.execute(verifyResultRequest, ApiUrlConstants.verifyResult, VerifyResultResponse.class);
    }

    /**
     * POST /api/file/sign/batch : 批量文件签署
     */
    public DssResponse<FileBatchSignResponse> fileSignBatch(FileBatchSignRequest batchSignRequest) throws ApiException {
        return clientService.execute(batchSignRequest, ApiUrlConstants.fileSignBatch, FileBatchSignResponse.class);
    }

    /**
     * 查询文件签署记录
     */
    public DssResponse<FileSignListResponse> fileSignList(FileSignListRequest request) throws ApiException {
        return clientService.execute(request, ApiUrlConstants.fileSignList, FileSignListResponse.class);
    }

    /**
     * EUI-获取文件预览页
     */
    public DssResponse<FilePreviewUrlResponse> filePreviewUrl(FilePreviewUrlRequest previewUrlRequest) throws ApiException {
        return clientService.execute(previewUrlRequest, ApiUrlConstants.filePreviewUrl, FilePreviewUrlResponse.class);
    }

    public DssResponse<FileSignUrlResponse> fileSignUrl(FileSignUrlRequest request) throws ApiException {
        return clientService.execute(request, ApiUrlConstants.fileSignUrl, FileSignUrlResponse.class);
    }

    public DssResponse<FileSignVerifyResponse> fileSignVerify(FileSignVerifyRequest request) throws ApiException {
        return clientService.execute(request, ApiUrlConstants.fileSignVerify, FileSignVerifyResponse.class);
    }
    /**
     * POST /api/file/sign/result : EUI-查询文件签署结果
     * 接口说明
     * 特别说明：
     * 1、此接口将用于EUI文件签署场景下，通过transactionId查询文件签署结果。
     * 适用场景：
     * 1、适用于需要查询文件签署记录的场景。
     * 注意事项：
     * 1、通过transactionId可以查询出单次签署记录。
     * @param request
     * @return
     */
    public DssResponse<FileSignResultResponse> signResultQuery(SignRequestQueryRequest request) throws ApiException {
        return clientService.execute(request, ApiUrlConstants.signResultQuery, FileSignResultResponse.class);
    }
}
