package com.dss.sdk.config;

import org.apache.hc.client5.http.HttpRequestRetryStrategy;
import org.apache.hc.client5.http.impl.DefaultHttpRequestRetryStrategy;
import org.apache.hc.client5.http.utils.DateUtils;
import org.apache.hc.core5.annotation.Contract;
import org.apache.hc.core5.annotation.ThreadingBehavior;
import org.apache.hc.core5.concurrent.CancellableDependency;
import org.apache.hc.core5.http.*;
import org.apache.hc.core5.http.protocol.HttpContext;
import org.apache.hc.core5.util.Args;
import org.apache.hc.core5.util.TimeValue;

import javax.net.ssl.SSLException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.ConnectException;
import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * 自定义重试 SocketTimeoutException
 * <p>
 * DefaultHttpRequestRetryStrategy针对InterruptedIOException不会重试
 *
 * @version 2.0.0
 * @author: Fadada
 * @date: 2024/12/23
 */
@Contract(threading = ThreadingBehavior.STATELESS)
public class CustomHttpRequestRetryStrategy implements HttpRequestRetryStrategy {
    /**
     * Default instance of {@link DefaultHttpRequestRetryStrategy}.
     */
    public static final CustomHttpRequestRetryStrategy INSTANCE = new CustomHttpRequestRetryStrategy();
    /**
     * Maximum number of allowed retries
     */
    private final int maxRetries;
    /**
     * Retry interval between subsequent retries
     */
    private final TimeValue defaultRetryInterval;
    /**
     * Derived {@code IOExceptions} which shall not be retried
     */
    private final Set<Class<? extends IOException>> nonRetriableIOExceptionClasses;
    /**
     * HTTP status codes which shall be retried
     */
    private final Set<Integer> retriableCodes;

    protected CustomHttpRequestRetryStrategy(
            final int maxRetries,
            final TimeValue defaultRetryInterval,
            final Collection<Class<? extends IOException>> clazzes,
            final Collection<Integer> codes) {
        Args.notNegative(maxRetries, "maxRetries");
        Args.notNegative(defaultRetryInterval.getDuration(), "defaultRetryInterval");
        this.maxRetries = maxRetries;
        this.defaultRetryInterval = defaultRetryInterval;
        this.nonRetriableIOExceptionClasses = new HashSet<>(clazzes);
        this.retriableCodes = new HashSet<>(codes);
    }

    public CustomHttpRequestRetryStrategy(
            final int maxRetries,
            final TimeValue defaultRetryInterval) {
        this(maxRetries, defaultRetryInterval,
                Arrays.asList(
                        UnknownHostException.class,
                        ConnectException.class,
                        ConnectionClosedException.class,
                        NoRouteToHostException.class,
                        SSLException.class),
                Arrays.asList(
                        HttpStatus.SC_TOO_MANY_REQUESTS,
                        HttpStatus.SC_SERVICE_UNAVAILABLE));
    }

    public CustomHttpRequestRetryStrategy(final int maxRetries) {
        this(maxRetries, TimeValue.ofSeconds(1L));
    }

    public CustomHttpRequestRetryStrategy() {
        this(3, TimeValue.ofSeconds(1L));
    }

    @Override
    public boolean retryRequest(HttpRequest request, IOException exception, int execCount, HttpContext context) {
        Args.notNull(request, "request");
        Args.notNull(exception, "exception");
        if (execCount > this.maxRetries) {
            // Do not retry if over max retries
            return false;
        }
        // SocketTimeoutException
        if (exception instanceof InterruptedIOException ||
                exception instanceof NoHttpResponseException) {
            // Timeout or 服务端断开连接
            return true;
        }
        if (this.nonRetriableIOExceptionClasses.contains(exception.getClass())) {
            return false;
        }
        for (final Class<? extends IOException> rejectException : this.nonRetriableIOExceptionClasses) {
            if (rejectException.isInstance(exception)) {
                return false;
            }
        }
        if (request instanceof CancellableDependency && ((CancellableDependency) request).isCancelled()) {
            return false;
        }

        // Retry if the request is considered idempotent
        return handleAsIdempotent(request);
    }

    @Override
    public boolean retryRequest(HttpResponse response, int execCount, HttpContext context) {
        Args.notNull(response, "response");
        return execCount <= this.maxRetries && retriableCodes.contains(response.getCode());
    }

    @Override
    public TimeValue getRetryInterval(HttpResponse response, int execCount, HttpContext context) {
        Args.notNull(response, "response");

        final Header header = response.getFirstHeader(HttpHeaders.RETRY_AFTER);
        TimeValue retryAfter = null;
        if (header != null) {
            final String value = header.getValue();
            try {
                retryAfter = TimeValue.ofSeconds(Long.parseLong(value));
            } catch (final NumberFormatException ignore) {
                final Instant retryAfterDate = DateUtils.parseStandardDate(value);
                if (retryAfterDate != null) {
                    retryAfter =
                            TimeValue.ofMilliseconds(retryAfterDate.toEpochMilli() - System.currentTimeMillis());
                }
            }

            if (TimeValue.isPositive(retryAfter)) {
                return retryAfter;
            }
        }
        return this.defaultRetryInterval;
    }

    protected boolean handleAsIdempotent(final HttpRequest request) {
        return Method.isIdempotent(request.getMethod());
    }
}
