package com.dss.sdk.file;

import com.dss.sdk.constants.Constants;
import com.dss.sdk.exception.ApiException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;

/**
 * 文件包装类，支持本地文件、字节数组和输入流三种方式。
 *
 * @author Fadada
 */
public class FileItem {
    private static final Logger log = LoggerFactory.getLogger(FileItem.class);

    private DssFile dssFile;

    /**
     * 基于本地文件的构造器，适用于上传本地文件。
     *
     * @param file 本地文件
     */
    public FileItem(final File file) {
        this.dssFile = new LocalContract(file);
    }

    /**
     * 基于文件绝对路径的构造器，适用于上传本地文件。
     *
     * @param filePath 文件绝对路径
     */
    public FileItem(String filePath) {
        this(new File(filePath));
    }

    /**
     * 基于文件名和字节数组的构造器。
     *
     * @param fileName 文件名
     * @param content  文件字节数组
     */
    public FileItem(String fileName, byte[] content) {
        this(fileName, content, null);
    }

    /**
     * 基于文件名、字节数组和媒体类型的构造器。
     *
     * @param fileName 文件名
     * @param content  文件字节数组
     * @param mimeType 媒体类型，如：image/jpeg, text/plain
     */
    public FileItem(String fileName, byte[] content, String mimeType) {
        this.dssFile = new ByteArrayContract(fileName, content, mimeType);
    }

    /**
     * 基于文件名和字节流的构造器，适应于全流式上传，减少本地内存开销。
     *
     * @param fileName 文件名
     * @param stream   文件字节流
     */
    public FileItem(String fileName, InputStream stream) {
        this(fileName, stream, null);
    }

    /**
     * 基于文件名、字节流和媒体类型的构造器，适应于全流式上传，减少本地内存开销。
     *
     * @param fileName 文件名
     * @param stream   文件字节流
     * @param mimeType 媒体类型，如：image/jpeg, text/plain
     */
    public FileItem(String fileName, InputStream stream, String mimeType) {
        this.dssFile = new StreamContract(fileName, stream, mimeType);
    }

    public boolean isValid() {
        return this.dssFile.isValid();
    }

    public String getFileName() {
        return this.dssFile.getFileName();
    }

    public String getMimeType() throws IOException {
        return this.dssFile.getMimeType();
    }

    public long getFileLength() {
        return this.dssFile.getFileLength();
    }

    public void write(OutputStream output) throws IOException {
        this.dssFile.write(output);
    }

    public byte[] getFile() throws ApiException {
        return this.dssFile.getFile();
    }

    private interface DssFile {
        boolean isValid();

        String getFileName();

        String getMimeType();

        long getFileLength();

        byte[] getFile() throws ApiException;

        void write(OutputStream output) throws IOException;
    }

    private static class LocalContract implements DssFile {
        private File file;

        public LocalContract(File file) {
            this.file = file;
        }

        @Override
        public boolean isValid() {
            return this.file != null && this.file.exists() && this.file.isFile();
        }

        @Override
        public String getFileName() {
            return this.file.getName();
        }

        @Override
        public String getMimeType() {
            return Constants.MIME_TYPE_DEFAULT;
        }

        @Override
        public long getFileLength() {
            return this.file.length();
        }

        @Override
        public byte[] getFile() throws ApiException {
            if (this.getFileLength() == 0) {
                return new byte[0];
            }
            FileInputStream inputStream = null;
            try {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                inputStream = new FileInputStream(file);
                byte[] b = new byte[(int) file.length()];
                int n;
                while ((n = inputStream.read(b)) != -1) {
                    bos.write(b, 0, n);
                }
                bos.close();
                return bos.toByteArray();
            } catch (IOException e) {
                log.error("Read FileItem Exception", e);
                throw new ApiException(e);
            } finally {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (Throwable e) {
                        log.error("Close Read FileItem Throwable", e);
                        throw new ApiException(e);
                    }
                }
            }
        }

        @Override
        public void write(OutputStream output) throws IOException {
            InputStream input = null;
            try {
                input = new FileInputStream(this.file);
                byte[] buffer = new byte[Constants.READ_BUFFER_SIZE];
                int n = 0;
                while (-1 != (n = input.read(buffer))) {
                    output.write(buffer, 0, n);
                }
            } finally {
                if (input != null) {
                    input.close();
                }
            }
        }
    }

    private static class ByteArrayContract implements DssFile {
        private String fileName;
        private byte[] content;
        private String mimeType;

        public ByteArrayContract(String fileName, byte[] content, String mimeType) {
            this.fileName = fileName;
            this.content = content;
            this.mimeType = mimeType;
        }

        @Override
        public boolean isValid() {
            return this.content != null && this.fileName != null;
        }

        @Override
        public String getFileName() {
            return this.fileName;
        }

        @Override
        public String getMimeType() {
            if (this.mimeType == null) {
                return Constants.MIME_TYPE_DEFAULT;
            } else {
                return this.mimeType;
            }
        }

        @Override
        public long getFileLength() {
            return this.content.length;
        }

        @Override
        public byte[] getFile() throws ApiException {
            return this.content;
        }

        @Override
        public void write(OutputStream output) throws IOException {
            output.write(this.content);
        }
    }

    private static class StreamContract implements DssFile {
        private String fileName;
        private InputStream stream;
        private String mimeType;

        public StreamContract(String fileName, InputStream stream, String mimeType) {
            this.fileName = fileName;
            this.stream = stream;
            this.mimeType = mimeType;
        }

        @Override
        public boolean isValid() {
            return this.stream != null && this.fileName != null;
        }

        @Override
        public String getFileName() {
            return this.fileName;
        }

        @Override
        public String getMimeType() {
            if (this.mimeType == null) {
                return Constants.MIME_TYPE_DEFAULT;
            } else {
                return this.mimeType;
            }
        }

        @Override
        public long getFileLength() {
            return 0L;
        }

        @Override
        public byte[] getFile() throws ApiException {
            try {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                byte[] b = new byte[Constants.READ_BUFFER_SIZE];
                int n;
                while ((n = stream.read(b)) != -1) {
                    bos.write(b, 0, n);
                }
                bos.close();
                return bos.toByteArray();
            } catch (IOException e) {
                log.error("Read FileItem Exception", e);
                throw new ApiException(e);
            } finally {
                if (stream != null) {
                    try {
                        stream.close();
                    } catch (Throwable e) {
                        log.error("Close Read FileItem Throwable", e);
                        throw new ApiException(e);
                    }
                }
            }
        }

        @Override
        public void write(OutputStream output) throws IOException {
            try {
                byte[] buffer = new byte[Constants.READ_BUFFER_SIZE];
                int n = 0;
                while (-1 != (n = stream.read(buffer))) {
                    output.write(buffer, 0, n);
                }
            } finally {
                if (stream != null) {
                    stream.close();
                }
            }
        }
    }

}
