package com.dss.sdk.utils.json;

import com.dss.sdk.constants.Constants;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

public class GsonUtil {
    private static final Logger log = LoggerFactory.getLogger(GsonUtil.class);
    private static final Gson gson;

    //判断gson对象是否存在了,不存在则创建对象
    static {
        String dataFormat = Constants.DATE_TIME_FORMAT;
        IntOrLongOrDoubleStrategy intOrLongOrDoubleStrategy = new IntOrLongOrDoubleStrategy();
        //当使用GsonBuilder方式时属性为空的时候输出来的json字符串是有键值key的,显示形式是"key":null，而直接new出来的就没有"key":null的
        gson = new GsonBuilder().disableHtmlEscaping()
                // 优先转换为Integer，如果转换失败再转换为Long，如果再次转换失败再转换为Double
                .setNumberToNumberStrategy(intOrLongOrDoubleStrategy)
                .setObjectToNumberStrategy(intOrLongOrDoubleStrategy)
                .setDateFormat(dataFormat)
                .create();
    }

    //无参的私有构造方法
    private GsonUtil() {
    }


    public static Gson getGson() {
        return gson;
    }

    /**
     * 将对象转成json格式
     *
     * @param object
     * @return String
     */
    public static String toJSONString(Object object) {
        return gson.toJson(object);
    }

    //toJsonStr
    public static String toJsonStr(Object object) {
        return toJSONString(object);
    }

    /**
     * 将json转成特定的cls的对象
     *
     * @param gsonString
     * @param cls
     * @return
     */
    public static <T> T fromJson(String gsonString, Class<T> cls) {
        return gson.fromJson(gsonString, cls);
    }

    public static <T> T fromJson(JsonObject jsonObject, Class<T> cls) {
        //传入json对象和对象类型,将json转成对象
        return gson.fromJson(jsonObject, cls);
    }

    public static <T> T fromJson(String json, Type typeOfT) {
        return gson.fromJson(json, typeOfT);
    }

    public static <T> T toBean(String jsonStr, final ParameterizedType parameterizedType) {
        if (jsonStr == null) {
            return null;
        }
        return gson.fromJson(jsonStr, parameterizedType);
    }

    //toBean
    public static <T> T toBean(String jsonStr, Class<T> cls) {
        return fromJson(jsonStr, cls);
    }

    public static JsonObject beanToJsonObj(Object bean) {
        return gson.toJsonTree(bean).getAsJsonObject();
    }


    /**
     * json字符串转成list
     *
     * @param json
     * @param cls
     * @return
     */
    public static <T> List<T> toList(String json, Class<T> cls) {
        return gson.fromJson(json, TypeToken.getParameterized(List.class, cls).getType());
    }


    /**
     * json字符串转成map
     *
     * @param gsonString
     * @return
     */
    public static Map<String, Object> toMap(String gsonString) {
        return toMap(gsonString, String.class, Object.class);
    }

    /**
     * json字符串转成map
     *
     * @param gsonString
     * @return
     */
    public static <V> Map<String, V> toMap(String gsonString, Class<V> valueCls) {
        return toMap(gsonString, String.class, valueCls);
    }

    /**
     * json字符串转成map
     *
     * @param gsonString
     * @param keyCls     key的类型
     * @param valueCls   value的类型
     * @return
     */
    public static <K, V> Map<K, V> toMap(String gsonString, Class<K> keyCls, Class<V> valueCls) {
        return gson.fromJson(gsonString, TypeToken.getParameterized(Map.class, keyCls, valueCls).getType());
    }
}