package com.hiiso.bridge.dataimpl.config;

import com.hiiso.bridge.tools.util.StrUtil;
import com.alibaba.druid.filter.config.ConfigTools;
import com.hiiso.bridge.data.IBridgeDataService;
import com.hiiso.bridge.dataimpl.core.BridgeS3DataServiceImpl;
import com.hiiso.bridge.s3.amazon.AmazonS3FactoryService;
import com.hiiso.bridge.s3.config.S3FileConfig;
import com.hiiso.bridge.s3.core.S3Factory;
import com.hiiso.bridge.s3.huawei.HuaweiS3FactoryService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

@Configuration
@ConditionalOnProperty(name = "file.storage.mode", havingValue = "s3", matchIfMissing = true)
@Slf4j
public class BridgeS3Config {

    @Value("${file.storage.s3.sdk:huawei}")
    private String sdk;

    @Value("${file.storage.s3.url:}")
    private String url;

    @Value("${file.storage.s3.root-path:}")
    private String rootPath;

    @Value("${file.storage.s3.bucket-name:}")
    private String bucketName;

    @Value("${file.storage.s3.secret-key:}")
    private String secretKey;

    @Value("${file.storage.s3.access-key:}")
    private String accessKey;

    @Value("${file.storage.s3.protocol:}")
    private String protocol;

    @Value("${file.storage.s3.amazon.signing-region:}")
    private String signingRegion;

    @Value("${file.storage.s3.huawei.security-token:}")
    private String securityToken;

    @Value("${file.storage.s3.config.decrypt:false}")
    private boolean decrypt;

    @Value("${file.storage.s3.config.decrypt.key.secret-key:}")
    private String secretDecryptKey;

    @Value("${file.storage.s3.config.decrypt.key.access-key:}")
    private String accessDecryptKey;

    @ConditionalOnMissingBean(IBridgeDataService.class)
    @Bean
    public IBridgeDataService bridgeDataService() throws Exception {
        log.info("数据存储模式：s3");
        switch (sdk) {
            case "huawei":
                S3Factory.setFactoryService(new HuaweiS3FactoryService());
                break;
            case "amazon":
                S3Factory.setFactoryService(new AmazonS3FactoryService());
                break;
            default:
                throw new RuntimeException("不支持sdk:" + sdk);
        }
        S3FileConfig s3FileConfig = new S3FileConfig();
        s3FileConfig.setBucketName(bucketName);
        s3FileConfig.setProtocol(protocol);
        if (decrypt) {
            if (StrUtil.isBlank(accessDecryptKey)) {
                s3FileConfig.setAccessKey(ConfigTools.decrypt(accessKey));
            } else {
                s3FileConfig.setAccessKey(ConfigTools.decrypt(accessDecryptKey, accessKey));
            }
            if (StrUtil.isBlank(secretDecryptKey)) {
                s3FileConfig.setAccessKey(ConfigTools.decrypt(secretKey));
            } else {
                s3FileConfig.setSecretKey(ConfigTools.decrypt(secretDecryptKey, secretKey));
            }
        } else {
            s3FileConfig.setAccessKey(accessKey);
            s3FileConfig.setSecretKey(secretKey);
        }
        s3FileConfig.setSecurityToken(securityToken);
        s3FileConfig.setSigningRegion(signingRegion);
        s3FileConfig.setUrl(url);
        s3FileConfig.check();
        return new BridgeS3DataServiceImpl(S3Factory.createFileClient(s3FileConfig), rootPath);
    }

}
