package com.hiiso.bridge.dataimpl.core;

import com.hiiso.bridge.tools.collection.CollUtil;
import com.hiiso.bridge.tools.io.IoUtil;
import com.hiiso.bridge.tools.util.StrUtil;
import com.hiiso.bridge.data.IBridgeDataService;
import com.hiiso.bridge.data.entity.BridgeFile;
import com.hiiso.bridge.data.entity.BridgeFileAttribute;
import com.hiiso.bridge.data.util.PathNameUtil;
import com.hiiso.bridge.s3.core.S3BucketClientService;
import com.hiiso.bridge.s3.core.S3File;
import com.hiiso.bridge.s3.core.S3Obj;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

public class BridgeS3DataServiceImpl implements IBridgeDataService {

    private S3BucketClientService client;

    private String rootPath;

    public BridgeS3DataServiceImpl(S3BucketClientService client, String rootPath) {
        this.client = client;
        this.rootPath = rootPath;
    }

    private String getPath(BridgeFile bridgeFile) {
        if (StrUtil.isBlank(rootPath)) {
            return bridgeFile.getPathname();
        }
        return new BridgeFile(rootPath, bridgeFile.getPathname()).getPathname();
    }

    @Override
    public void writeBytes(byte[] data, BridgeFile file) {
        client.write(getPath(file), data);
    }

    @Override
    public void writeFromStream(InputStream inputStream, BridgeFile file) {
        byte[] bytes = IoUtil.readBytes(inputStream);
        writeBytes(bytes, file);
    }

    @Override
    public void writeFromStream(InputStream inputStream, BridgeFile file, long fileSize) {
        client.write(getPath(file), inputStream, fileSize);
    }

    @Override
    public byte[] readBytes(BridgeFile file) {
        return client.read(getPath(file));
    }

    @Override
    public String readString(BridgeFile file, Charset charset) {
        return client.readStr(getPath(file), charset);
    }

    @Override
    public InputStream getInputStream(BridgeFile file) {
        return client.readStream(getPath(file));
    }

    @Override
    public boolean exist(BridgeFile file) {
        return client.exist(getPath(file));
    }

    @Override
    public void delete(BridgeFile file) {
        String path = getPath(file);
        if (client.isFile(path)) {
            client.deleteFile(path);
        } else {
            client.del(path);
        }
    }

    @Override
    public List<BridgeFile> list(BridgeFile file) {
        String filePath = getPath(file);
        List<S3Obj> objs = client.list(filePath);
        if (CollUtil.isEmpty(objs)) {
            return null;
        }
        List<BridgeFile> list = new ArrayList<>();
        for (S3Obj obj : objs) {
            list.add(new BridgeFile(PathNameUtil.getRelativePath(obj.getPath(), rootPath)));
        }
        return list;
    }

    @Override
    public BridgeFileAttribute attr(BridgeFile file) {
        String filePath = getPath(file);
        S3File s3File = client.newFile(filePath);
        BridgeFileAttribute attribute = new BridgeFileAttribute();
        attribute.setLastModified(s3File.getLastModified());
        attribute.setSize(s3File.getSize());
        return attribute;
    }

    @Override
    public void rename(BridgeFile file, String newName, boolean isRetainExt, boolean isOverride) {
        client.rename(getPath(file), newName, isRetainExt, isOverride);
    }

    @Override
    public void copy(BridgeFile source, BridgeFile target, boolean isOverride) {
        client.copy(getPath(source), getPath(target), isOverride);
    }

    @Override
    public boolean isDirectory(BridgeFile file) {
        return client.isDirectory(getPath(file));
    }

    @Override
    public boolean isFile(BridgeFile file) {
        return client.isFile(getPath(file));
    }

    @Override
    public void mkdir(BridgeFile file) {
    }

    @Override
    public void copyContent(BridgeFile source, BridgeFile target, boolean isOverride) {
        client.copyContent(getPath(source), getPath(target), isOverride);
    }
}
