package com.github.hepeng86.mybatisplus.encrypt.core;

import com.baomidou.mybatisplus.annotation.TableName;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.enums.SqlKeyword;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Assert;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.LambdaUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.sql.StringEscape;
import com.baomidou.mybatisplus.core.toolkit.support.SFunction;
import com.baomidou.mybatisplus.core.toolkit.support.SerializedLambda;
import com.github.hepeng86.mybatisplus.encrypt.Encrypt;
import com.github.hepeng86.mybatisplus.encrypt.annotation.EncryptClass;
import com.github.hepeng86.mybatisplus.encrypt.annotation.EncryptField;
import com.github.hepeng86.mybatisplus.encrypt.constant.EncryptConstants;
import com.github.hepeng86.mybatisplus.encrypt.constant.MySQLKeyword;
import com.github.hepeng86.mybatisplus.encrypt.model.SensitiveField;
import com.github.hepeng86.mybatisplus.encrypt.model.TableInfo;
import com.github.hepeng86.mybatisplus.encrypt.util.ReflectionUtils;
import com.github.hepeng86.mybatisplus.encrypt.util.SpringBeanUtil;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.reflection.property.PropertyNamer;

@Slf4j
public final class CleanTableProcessor extends AbstractFieldConvertor {

    private final Encrypt encrypt;

    public CleanTableProcessor(Encrypt encrypt) {
        this.encrypt = encrypt;
    }

    public <T> void cleanTable(TableInfo<T> tableInfo) throws NoSuchFieldException, IllegalAccessException {
        Class<? extends BaseMapper<T>> mapperClass = tableInfo.getMapperClass();
        Assert.notNull(mapperClass, "mapperClass must not be null");

        BaseMapper<T> baseMapper = SpringBeanUtil.getBean(mapperClass);
        Assert.notNull(mapperClass, "The mapperClass '%s' bean not found", mapperClass);

        Class<T> entityClass = tableInfo.getEntityClass();
        Assert.notNull(entityClass, "entityClass must not be null");

        EncryptClass encryptClass = entityClass.getAnnotation(EncryptClass.class);
        Assert.notNull(encryptClass, "The entityClass '%s' is not @EncryptClass annotation", entityClass);

        SFunction<T, String>[] sensitiveColumns = tableInfo.getSensitiveColumns();
        Assert.notEmpty(sensitiveColumns, "sensitiveColumns must not be null or empty");

        List<String> normalColumnNameList = new ArrayList<>();
        for (SFunction<T, String> sensitiveColumn : sensitiveColumns) {
            Field sensitiveField = ReflectionUtils.getFieldFromCache(entityClass, getFieldNameBySFunction(sensitiveColumn));
            EncryptField encryptField = sensitiveField.getAnnotation(EncryptField.class);
            Assert.notNull(encryptField, "The fieldName '%s' is not @EncryptField annotation", sensitiveField.getName());

            normalColumnNameList.add(ReflectionUtils.getColumnNameByField(sensitiveField));
        }

        TableName tableNameAnnotation = entityClass.getAnnotation(TableName.class);
        String tableName = (Objects.isNull(tableNameAnnotation) || StringUtils.isBlank(tableNameAnnotation.value())) ? StringUtils.camelToUnderline(entityClass.getSimpleName()) : tableNameAnnotation.value();

        log.info("Currently cleaning tableName:{} and columnNames:{}", tableName.toUpperCase(), normalColumnNameList);

        SFunction<T, Long> primaryColumn = tableInfo.getPrimaryColumn();
        String primaryColumnName;
        if (Objects.isNull(primaryColumn)) {
            primaryColumnName = "ID";
        } else {
            Field primaryField = ReflectionUtils.getFieldFromCache(entityClass, getFieldNameBySFunction(primaryColumn));
            primaryColumnName = ReflectionUtils.getColumnNameByField(primaryField);
        }

        String normalColumnNames = String.join(",", normalColumnNameList);

        Map<String, SensitiveField> columnNameAndFieldMap = ReflectionUtils.getEncryptColumnNameAndFieldMapFromCache(tableInfo.getEntityClass());

        long retryId = 0;
        long id = 0;
        outerLoop:
        for (;;) {
            QueryWrapper<T> queryWrapper = new QueryWrapper<>();
            queryWrapper.setEntityClass(tableInfo.getEntityClass());
            queryWrapper.select(primaryColumnName, normalColumnNames);
            queryWrapper.gt(primaryColumnName, id);
            // Compatible Query Logic Deletion
            queryWrapper.last(String.format("%s %s %s %s %s %s", SqlKeyword.OR.getSqlSegment(), primaryColumnName, SqlKeyword.GT.getSqlSegment(), id, MySQLKeyword.LIMIT.getSqlSegment(), EncryptConstants.PAGE_SIZE));

            List<T> list = baseMapper.selectList(queryWrapper);
            if (CollectionUtils.isEmpty(list)) {
                break;
            }

            for (T entity : list) {
                id = (Long)ReflectionUtils.getValue(entity, StringUtils.underlineToCamel(primaryColumnName));

                Map<String, String> nonBlankColumnMap = getNonBlankColumnMap(entity, normalColumnNameList);
                if (CollectionUtils.isEmpty(nonBlankColumnMap)) {
                    continue;
                }

                UpdateWrapper<T> updateWrapper = new UpdateWrapper<>();
                updateWrapper.eq(primaryColumnName, id);

                StringBuilder sqlSegment = new StringBuilder();
                sqlSegment.append(String.format(" %s %s %s %s", SqlKeyword.OR.getSqlSegment(), primaryColumnName, SqlKeyword.EQ.getSqlSegment(), id));

                Set<Entry<String, String>> entries = nonBlankColumnMap.entrySet();
                for (Entry<String, String> entry : entries) {
                    SensitiveField field = columnNameAndFieldMap.get(entry.getKey());
                    String encryptValue = convert(entry.getValue(), field.getJsonPaths());
                    updateWrapper.in(entry.getKey(), entry.getValue(), encryptValue);

                    sqlSegment.append(String.format(" %s %s %s (%s, %s)", SqlKeyword.AND.getSqlSegment(), entry.getKey(), SqlKeyword.IN.getSqlSegment(), StringEscape.escapeString(entry.getValue()), StringEscape.escapeString(encryptValue)));
                }

                // Compatible Update for Logical Deletion
                updateWrapper.last(sqlSegment.toString());

                int updatedCnt = baseMapper.update(entity, updateWrapper);
                if (updatedCnt == 0) {
                    if (retryId != id) {
                        retryId = id;
                        id --;
                        continue outerLoop;
                    } else {
                        log.warn("Currently cleaning tableName:{} [{}:{}] update fail", tableName.toUpperCase(), primaryColumnName, id);
                    }
                }
                retryId = 0;
            }

            if (list.size() < EncryptConstants.PAGE_SIZE) {
                break;
            }
        }
    }

    private static <T> String getFieldNameBySFunction(SFunction<T, ?> sFunction) {
        SerializedLambda lambda = LambdaUtils.resolve(sFunction);
        return PropertyNamer.methodToProperty(lambda.getImplMethodName());
    }

    private static <T> Map<String, String> getNonBlankColumnMap(T entity, List<String> columnNameList) throws NoSuchFieldException, IllegalAccessException {
        Map<String, String> nonBlankColumnMap = new HashMap<>();
        for (String columnName : columnNameList) {
            String columnValue = (String)ReflectionUtils.getValue(entity, StringUtils.underlineToCamel(columnName));
            if (StringUtils.isNotBlank(columnValue)) {
                nonBlankColumnMap.put(columnName, columnValue);
            }
        }

        return nonBlankColumnMap;
    }

    @Override
    protected String convert(String value) {
        return encrypt.encrypt(value);
    }
}
