package com.github.hepeng86.mybatisplus.encrypt.plugin;

import com.baomidou.mybatisplus.core.conditions.AbstractWrapper;
import com.baomidou.mybatisplus.core.conditions.ISqlSegment;
import com.baomidou.mybatisplus.core.conditions.segments.MergeSegments;
import com.baomidou.mybatisplus.core.conditions.segments.NormalSegmentList;
import com.baomidou.mybatisplus.core.enums.SqlKeyword;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.github.hepeng86.mybatisplus.encrypt.Encrypt;
import com.github.hepeng86.mybatisplus.encrypt.core.AbstractFieldConvertor;
import com.github.hepeng86.mybatisplus.encrypt.properties.EncryptConfigProperties;
import com.github.hepeng86.mybatisplus.encrypt.constant.EncryptConstants;
import com.github.hepeng86.mybatisplus.encrypt.util.ReflectionUtils;
import com.github.hepeng86.mybatisplus.encrypt.model.SensitiveField;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.plugin.Interceptor;
import org.springframework.util.Assert;

public abstract class AbstractInterceptor extends AbstractFieldConvertor implements Interceptor {

    protected final Encrypt encrypt;

    protected final EncryptConfigProperties encryptConfigProperties;

    public AbstractInterceptor(Encrypt encrypt, EncryptConfigProperties encryptConfigProperties) {
        Objects.requireNonNull(encrypt, "encrypt should not be null!");
        String plaintext = "Encryption and Decryption Algorithm Preheating...";
        String encryptText = encrypt.encrypt(plaintext);
        String decryptText = encrypt.decrypt(encryptText);
        Assert.isTrue(plaintext.equals(decryptText), "Inconsistency in Encryption and Decryption Algorithms");

        this.encrypt = encrypt;
        this.encryptConfigProperties = encryptConfigProperties;
    }

    protected void handleParameters(Object param) throws Exception {
        Map<String, SensitiveField> columnNameAndFieldMap = ReflectionUtils.getEncryptColumnNameAndFieldMapFromCache(param.getClass());
        if (CollectionUtils.isEmpty(columnNameAndFieldMap)) {
            return;
        }

        Set<Entry<String, SensitiveField>> entries = columnNameAndFieldMap.entrySet();
        for (Entry<String, SensitiveField> entry : entries) {
            SensitiveField field = entry.getValue();

            String value = field.getValue(param);
            if (StringUtils.isBlank(value)) {
                continue;
            }

            String encryptValue = convert(value, field.getJsonPaths());
            field.setValue(param, encryptValue);
        }
    }

    protected void handleParameters(AbstractWrapper<?, ?, ?> wrapper, BoundSql boundSql) throws Exception {
        AtomicInteger seq = getSeq(wrapper);
        if (seq.get() == -1) {
            return;
        }
        handleParameters(wrapper, seq, boundSql);
    }

    protected AtomicInteger getSeq(AbstractWrapper<?, ?, ?> wrapper) {
        MergeSegments mergeSegments = wrapper.getExpression();
        String sqlSegment = mergeSegments.getSqlSegment();
        Matcher matcher = EncryptConstants.VAL_PATTERN.matcher(sqlSegment);
        int seq = matcher.find() ? Integer.parseInt(matcher.group(1)) : -1;
        return new AtomicInteger(seq);
    }

    protected void handleParameters(AbstractWrapper<?, ?, ?> wrapper, AtomicInteger seq, BoundSql boundSql) throws Exception {
        SqlKeyword sqlKeyword = null;
        String columnName = null;
        int ignoreSteps = 0;
        MergeSegments mergeSegments = wrapper.getExpression();
        NormalSegmentList normalSegmentList = mergeSegments.getNormal();

        for (ISqlSegment iSqlSegment : normalSegmentList) {
            if (iSqlSegment instanceof SqlKeyword) {
                sqlKeyword = (SqlKeyword) iSqlSegment;
                if (Arrays.asList(SqlKeyword.BETWEEN, SqlKeyword.NOT_BETWEEN).contains(sqlKeyword)) {
                    ignoreSteps = 2;
                }
                continue;
            }

            if (iSqlSegment instanceof AbstractWrapper) {
                handleParameters((AbstractWrapper<?, ?, ?>) iSqlSegment, seq, boundSql);
                continue;
            }

            if (Objects.isNull(columnName)) {
                columnName = iSqlSegment.getSqlSegment();
                continue;
            }

            int curSeq = seq.getAndIncrement();
            String curColumnName = columnName;
            columnName = null;

            if (ignoreSteps > 0) {
                ignoreSteps--;
                continue;
            }

            SensitiveField sensitiveField = ReflectionUtils.getSensitiveField(wrapper.getEntityClass(), curColumnName);
            if (Objects.isNull(sensitiveField)) {
                continue;
            }

            Object v2 = ReflectionUtils.getValue(iSqlSegment, "arg$2");
            if (isNullOrBlank(v2)) {
                continue;
            }

            if (v2 instanceof List) {
                List<?> values = (List<?>) v2;
                for (int i = 0, size = values.size(); i < size; i++) {
                    Object v21 = values.get(i);
                    if (i > 0) {
                        curSeq = seq.getAndIncrement();
                    }

                    if (isNullOrBlank(v21)) {
                        continue;
                    }

                    setParamNameValuePairs(curSeq, (String)v21, wrapper, sensitiveField, boundSql, sqlKeyword);
                }
            } else {
                setParamNameValuePairs(curSeq, (String)v2, wrapper, sensitiveField, boundSql, sqlKeyword);
            }
        }
    }

    protected boolean isNullOrBlank(Object o) {
        return Objects.isNull(o) || (o instanceof String && StringUtils.isBlank((String)o));
    }

    protected void setParamNameValuePairs(int seq,
            String value,
            AbstractWrapper<?, ?, ?> wrapper,
            SensitiveField sensitiveField,
            BoundSql boundSql,
            SqlKeyword sqlKeyword) throws Exception {

    }

    protected boolean isMixedQueryMode() {
        return encryptConfigProperties.isMixedQueryMode();
    }
}
