package com.github.hepeng86.mybatisplus.encrypt.util;

import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.core.toolkit.Assert;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.github.hepeng86.mybatisplus.encrypt.annotation.EncryptClass;
import com.github.hepeng86.mybatisplus.encrypt.annotation.EncryptField;
import com.github.hepeng86.mybatisplus.encrypt.model.SensitiveField;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public final class ReflectionUtils {

    private static final Map<Class<?>, Map<String, SensitiveField>> COLUMN_NAME_AND_FIELD_MAP = new HashMap<>();

    private static final Map<Class<?>, Map<String, Field>> FIELD_NAME_AND_FIELD_MAP = new HashMap<>();

    public static SensitiveField getSensitiveField(Class<?> clz, String columnName) {
        return getEncryptColumnNameAndFieldMapFromCache(clz).get(formatColumnName(columnName));
    }

    public static Map<String, SensitiveField> getEncryptColumnNameAndFieldMapFromCache(Class<?> clz) {
        return COLUMN_NAME_AND_FIELD_MAP.computeIfAbsent(clz, aClass -> {
            EncryptClass encryptClass = clz.getAnnotation(EncryptClass.class);
            if (Objects.isNull(encryptClass)) {
                return Collections.emptyMap();
            }

            Map<String, SensitiveField> columnNameAndFieldMap = new HashMap<>();
            Field[] declaredFields = getAllDeclaredFields(aClass);
            for (Field field : declaredFields) {
                field.setAccessible(true);
                EncryptField encryptField = field.getAnnotation(EncryptField.class);
                if (Objects.isNull(encryptField)) {
                    continue;
                }

                Assert.isTrue(field.getType().equals(String.class), "The annotation @EncryptField is not supported on fields of non-String types. The expected field type is String.");

                TableField tableField = field.getAnnotation(TableField.class);
                String columnName = (Objects.isNull(tableField) || StringUtils.isBlank(tableField.value())) ? StringUtils.camelToUnderline(field.getName()) : tableField.value();
                columnNameAndFieldMap.put(formatColumnName(columnName), new SensitiveField(field, encryptField.jsonPaths()));
            }

            if (CollectionUtils.isEmpty(columnNameAndFieldMap)) {
                return Collections.emptyMap();
            }

            return Collections.unmodifiableMap(columnNameAndFieldMap);
        });
    }

    public static Field[] getAllDeclaredFields(Class<?> clazz) {
        List<Field> fields = new ArrayList<>();
        while (clazz != null) {
            Field[] declaredFields = clazz.getDeclaredFields();
            if (declaredFields.length > 0) {
                fields.addAll(Arrays.asList(declaredFields));
            }
            clazz = clazz.getSuperclass();
        }
        return fields.toArray(new Field[0]);
    }

    private static String formatColumnName(String columnName) {
        return columnName.toUpperCase();
    }

    public static void setField(Object target, String fieldName, Object value) throws IllegalAccessException, NoSuchFieldException {
        Field field = getFieldFromCache(target.getClass(), fieldName);
        field.set(target, value);
    }

    public static Object getValue(Object target, String fieldName) throws IllegalAccessException, NoSuchFieldException {
        Field field = getFieldFromCache(target.getClass(), fieldName);
        return field.get(target);
    }

    public static Field getFieldFromCache(Class<?> clz, String fieldName) throws NoSuchFieldException {
        Map<String, Field> fieldNameAndFieldMap = FIELD_NAME_AND_FIELD_MAP.computeIfAbsent(clz, aClass -> new HashMap<>());
        Field field = fieldNameAndFieldMap.get(fieldName);
        if (Objects.isNull(field)) {
            field = getDeclaredField(clz, fieldName);
            field.setAccessible(true);
            fieldNameAndFieldMap.put(fieldName, field);
        }
        return field;
    }

    public static Field getDeclaredField(Class<?> clazz, String fieldName) throws NoSuchFieldException {
        while (true) {
            try {
                return clazz.getDeclaredField(fieldName);
            } catch (NoSuchFieldException e) {
                clazz = clazz.getSuperclass();
                if (clazz == null) {
                    throw e;
                }
            }
        }
    }

    public static String getColumnNameByField(Field field) {
        TableField tableField = field.getAnnotation(TableField.class);
        String columnName = (Objects.isNull(tableField) || StringUtils.isBlank(tableField.value())) ? StringUtils.camelToUnderline(field.getName()) : tableField.value();
        return formatColumnName(columnName);
    }
}
