package cc.jq1024.middleware.token.service.impl;

import cc.jq1024.middleware.token.config.TokenProperties;
import cc.jq1024.middleware.token.service.ITokenService;
import cc.jq1024.middleware.token.service.TokenServiceSupport;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.HashMap;


/**
 * @author li--jiaqiang
 */
public class TokenService extends TokenServiceSupport implements ITokenService {

    private static final Logger log = LoggerFactory.getLogger(TokenService.class);

    public TokenService(TokenProperties tokenProperties) {
        super(tokenProperties);
    }

    /**
     * 这里就是产生jwt字符串的地方
     * jwt字符串包括三个部分
     * 1. header
     * -当前字符串的类型，一般都是“JWT”
     * -哪种算法加密，“HS256”或者其他的加密算法
     * 所以一般都是固定的，没有什么变化
     * 2. payload
     * 一般有四个最常见的标准字段（下面有）
     * iat：签发时间，也就是这个jwt什么时候生成的
     * jti：JWT的唯一标识
     * iss：签发人，一般都是username或者userId
     * exp：过期时间
     */
    @Override
    public String createJWT(HashMap<String, Object> claims) {
        return this.createJWT(getUUID(), claims, tokenProperties.getTtl());
    }

    @Override
    public String createJWT(String id, HashMap<String, Object> claims, Long ttlMillis) {
        // iss签发人，ttlMillis生存时间，claims是指还想要在jwt中存储的一些非隐私信息
        if (claims == null) {
            claims = new HashMap<>();
        }
        long nowMillis = System.currentTimeMillis();
        JwtBuilder builder = Jwts.builder()
                // 荷载部分
                .setClaims(claims)
                // 这个是JWT的唯一标识，一般设置成唯一的，这个方法可以生成唯一标识
                .setId(id)
                // 签发时间
                .setIssuedAt(new Date(nowMillis))
                // 签发人，也就是JWT是给谁的（逻辑上一般都是username或者userId）
                .setSubject(tokenProperties.getIssuer())
                .signWith(SignatureAlgorithm.HS256, encodeBase64SecretKey());//这个地方是生成jwt使用的算法和秘钥
        if (ttlMillis <= 0) {
            // 防止 ttl <= 0
            ttlMillis = tokenProperties.getTtl();
        }
        // 4. 过期时间，这个也是使用毫秒生成的，使用当前时间+前面传入的持续时间生成
        long expMillis = nowMillis + ttlMillis;
        Date exp = new Date(expMillis);
        builder.setExpiration(exp);
        return builder.compact();
    }

    @Override
    public String createRefreshJWT(HashMap<String, Object> claims) {
        return createJWT(getUUID(), claims, tokenProperties.getTtl() * 2);
    }

    @Override
    public String propertyValue(String token, String property) {
        try {
            Claims claims = decode(token);
            return claims.get(property).toString();
        } catch (Exception e) {
            log.error("解析 {} 出错: ", property, e);
            return null;
        }
    }

    // 相当于encode的方向，传入jwtToken生成对应的username和password等字段。Claim就是一个map
    // 也就是拿到荷载部分所有的键值对
    @Override
    public Claims decode(String jwtToken) {
        return Jwts.parser()
                // 设置签名的秘钥
                .setSigningKey(encodeBase64SecretKey())
                // 设置需要解析的 jwt
                .parseClaimsJws(jwtToken)
                .getBody();
    }

    // 判断jwtToken是否合法
    @Override
    public boolean isVerify(String jwtToken) {
        try {
            JWTVerifier verifier = JWT.require(encodeHS256SecretKey()).build();
            verifier.verify(jwtToken);
            // 校验不通过会抛出异常
            // 判断合法的标准：1. 头部和荷载部分没有篡改过。2. 没有过期
            return true;
        } catch (Exception e) {
            log.error("jwt verify Error", e);
            return false;
        }
    }

}