package io.gitee.open.nw.common.util;

import com.alibaba.fastjson2.JSON;
import io.gitee.open.nw.common.base.AppContext;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import java.io.*;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * http 请求工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class HttpUtil {

    public static void main(String[] args) {


        HashMap<Object, Object> objectObjectHashMap = new HashMap<>();
        objectObjectHashMap.put("a", "1");
        objectObjectHashMap.put("b", "2");
        String jsonString = StringUtil.replaceAllBlank(JSON.toJSONString(objectObjectHashMap));
        System.out.println(jsonString);

        String request = request("https://tooltt.com/json2typescript/");
    }

    private final static Charset CHARSET = StandardCharsets.UTF_8;

    private final static int CONNECT_TIME = 2000;

    private final static int READ_TIME = 90000;

    public static String request(String requestUrl, Boolean enableLog) {
        return request(HttpMethod.GET, requestUrl, null, null, null, null, enableLog);
    }

    public static String request(String requestUrl) {
        return request(HttpMethod.GET, requestUrl, null, null, null, null);
    }

    public static String request(String requestUrl, String body) {
        return request(HttpMethod.POST, requestUrl, body, ContentType.JSON);
    }

    public static String request(String requestUrl, String body, boolean enableLog) {
        return request(HttpMethod.POST, requestUrl, body, ContentType.JSON, enableLog);
    }

    public static String request(String requestUrl, String body, Logger enableLog) {
        return request(HttpMethod.POST, requestUrl, body, ContentType.JSON);
    }

    public static String request(HttpMethod method, String requestUrl, String body, ContentType contentType) {
        Map<String, String> headers = new HashMap<>(4);
        headers.put("Content-Type", contentType.value);
        return request(method, requestUrl, body, headers, null, null);
    }

    public static String request(HttpMethod method, String requestUrl, String body, ContentType contentType,
                                 boolean enableLog) {
        Map<String, String> headers = new HashMap<>(4);
        headers.put("Content-Type", contentType.value);
        return request(method, requestUrl, body, headers, null, null, enableLog);
    }

    public String request(HttpMethod method, String requestUrl, String body, String soupAction) {
        Map<String, String> headers = new HashMap<>(4);
        headers.put("Content-Type", ContentType.XML.value);
        headers.put("SOAPAction", soupAction);
        return request(method, requestUrl, body, headers, null, null);
    }

    public static String request(HttpMethod method, String requestUrl, String body, Map<String, String> headers, Integer readTimeout, Charset charset) {
        if (charset == null) {
            charset = CHARSET;
        }
        return request(method, requestUrl, body, headers, readTimeout, charset, true);
    }

    public static String request(HttpMethod method, String requestUrl, String body, Map<String, String> headers,
                                 Integer readTimeout, Charset charset, ResponseCallBack callBack) {
        return request(method, requestUrl, body, headers, readTimeout, charset, true);
    }

    public static String request(HttpMethod method, String requestUrl, String body, Map<String, String> headers,
                                 Integer readTimeout, Charset charset, boolean enableLog) {

        return new String(requestByteArray(method, requestUrl, body, headers, readTimeout, charset, enableLog), StandardCharsets.UTF_8);
    }

    public static byte[] requestByteArray(HttpMethod method, String requestUrl, String body, ContentType contentType) {
        Map<String, String> headers = new HashMap<>(4);
        headers.put("Content-Type", contentType.value);
        return requestByteArray(method, requestUrl, body, headers, null, null, true);
    }

    public static byte[] requestByteArray(HttpMethod method, String requestUrl, String body, Map<String, String> headers,
                                          Integer readTimeout, Charset charset,
                                          boolean enableLog) {
        if (charset == null) {
            charset = CHARSET;
        }

        if (enableLog) {
            AppContext.get(Logger.class).info("Request Start url: {} method:{}", requestUrl, method.toString());
            if (headers != null && !headers.isEmpty()) {
                AppContext.get(Logger.class).info("Request Header: {}", StringUtil.replaceAllBlank(JSON.toJSONString(headers)));
            }
            if (StringUtils.isNotEmpty(body)) {
                AppContext.get(Logger.class).info("Request Body: {}", body);
            }
        }
        HttpURLConnection conn = null;
        try {
            //创建和初始化连接
            for (int i = 0; i < 3; i++) {
                try {
                    conn = openConnection(method, requestUrl, body, headers, readTimeout, charset);
                    break;
                } catch (ConnectException e) {
                    AppContext.get(Logger.class).info("Connect Faild {} Retry Times {}", requestUrl, i + 1);
                }
            }
            if (conn == null) {
                throw new RuntimeException("Connect Faild Place Check");
            }
            try (InputStream in = conn.getInputStream(); ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
                byte[] buffer = new byte[1024];
                int len;
                while ((len = in.read(buffer)) != -1) {
                    bos.write(buffer, 0, len);
                }
                bos.flush();
                byte[] byteArray = bos.toByteArray();
                Map<String, List<String>> headerFields = conn.getHeaderFields();
                List<String> contentTypeList = headerFields.get("Content-Type");
                // 判断是不是文本
                if (enableLog && contentTypeList != null && !contentTypeList.isEmpty()) {
                    String contentType = contentTypeList.get(0);
                    String response = new String(byteArray, charset);
                    if (contentType.contains("text") || contentType.contains("xml") || contentType.contains("json")) {
                        AppContext.get(Logger.class).info("Request End Response: {}", response);
                    }
                }
                return byteArray;
            } catch (IOException e) {
                //处理错误流，提示错误信息
                try (InputStream es = conn.getErrorStream()) {
                    if (es != null) {
                        BufferedReader br = new BufferedReader(new InputStreamReader(es));
                        String line;
                        StringBuilder strBuilder = new StringBuilder();
                        while ((line = br.readLine()) != null) {
                            strBuilder.append(line);
                        }
                        AppContext.get(Logger.class).error("Request Faild！URL:{} body:{} info:{}", requestUrl, body, strBuilder);
                        throw new HttpException(conn.getResponseCode(), strBuilder.toString());
                    } else {
                        AppContext.get(Logger.class).error("Request Faild");
                        throw new HttpException(conn.getResponseCode(), "Request Faild");
                    }
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e.getMessage());
        } finally {
            //关闭连接
            if (conn != null) {
                conn.disconnect();
            }
        }
    }

    /**
     * 打开链接设置参数
     */
    public static HttpURLConnection openConnection(HttpMethod method, String requestUrl, String body, Map<String, String> headers, Integer readTimeout, Charset charset) throws IOException {
        if (method == null) {
            method = HttpMethod.GET;
        }
        if (charset == null) {
            charset = CHARSET;
        }
        if (readTimeout == null) {
            readTimeout = READ_TIME;
        }
        URL url = new URL(requestUrl);
        HttpURLConnection conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod(method.toString());
        conn.setDoOutput(true);
        conn.setDoInput(true);
        //设置连接超时时间
        conn.setConnectTimeout(CONNECT_TIME);
        //设置读取超时时间
        conn.setReadTimeout(readTimeout);
        //指定请求header参数
        if (headers != null && !headers.isEmpty()) {
            Set<Map.Entry<String, String>> entries = headers.entrySet();
            for (Map.Entry<String, String> entry : entries) {
                conn.setRequestProperty(entry.getKey(), entry.getValue());
            }
        }
        if (!Objects.equals(body, "") && body != null) {
            OutputStream outputStream = conn.getOutputStream();
            outputStream.write(body.getBytes(charset));
            outputStream.flush();
        }
        return conn;
    }

    /**
     * 支持的Http method
     */
    public enum HttpMethod {
        // post请求
        POST,
        // delete请求
        DELETE,
        // get请求
        GET,
        // put请求
        PUT,
        // head请求
        HEAD;
    }

    /**
     * 支持的ContentType
     */
    public enum ContentType {
        JSON("application/json;charset=utf-8"),
        XML("text/xml;charset=utf-8"),
        FORM("application/x-www-form-urlencoded");
        final String value;

        public String getValue() {
            return value;
        }

        ContentType(String value) {
            this.value = value;
        }
    }

    @FunctionalInterface
    public interface ResponseCallBack {
        /**
         * 回调
         */
        void callBack(InputStream inputStream);
    }

    public static class HttpException extends RuntimeException {

        private final Integer httpCode;


        public HttpException(Integer httpCode, String message) {
            super(message);
            this.httpCode = httpCode;
        }

        public Integer getHttpCode() {
            return httpCode;
        }
    }
}
