package io.gitee.open.nw.common.util;

import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.RequestMethod;

import java.io.ByteArrayInputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.*;

/**
 * 微信工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class WeiXinUtil {

    public static String requestWxPay(HttpUtil.HttpMethod method, String url, String body, String wxMchId, String wxMchSerialNo, String wxOriginalKey) throws Exception {
        String authorization = buildAuthorization(method.name(), url.replaceAll("https://api.mch.weixin.qq.com", ""), wxMchId, wxMchSerialNo, wxOriginalKey, body);
        HashMap<String, String> headers = new HashMap<>();
        headers.put("Authorization", authorization);
        headers.put("Content-Type", "application/json");
        headers.put("Accept", "application/json");
        return HttpUtil.request(method, url, body, headers, null, null);
    }

    private static final String KEY_ALGORITHM = "RSA";


    /**
     * 构建 v3 接口所需的 Authorization
     *
     * @param method      请求方法
     * @param urlSuffix   可通过 WxApiType 来获取，URL挂载参数需要自行拼接
     * @param mchId       商户Id
     * @param serialNo    商户 API 证书序列号
     * @param originalKey key.pem 证书字符串
     * @param body        接口请求参数
     * @return {@link String} 返回 v3 所需的 Authorization
     * @throws Exception 异常信息
     */
    public static String buildAuthorization(String method, String urlSuffix, String mchId,
                                            String serialNo, String originalKey, String body) throws Exception {

        long timestamp = System.currentTimeMillis() / 1000;
        String authType = "WECHATPAY2-SHA256-RSA2048";
        String nonceStr = UUID.randomUUID().toString().replaceAll("-", "");
        return buildAuthorization(method, urlSuffix, mchId, serialNo, originalKey, body, nonceStr, timestamp, authType);
    }


    /**
     * 构建 v3 接口所需的 Authorization
     *
     * @param method      {@link RequestMethod} 请求方法
     * @param urlSuffix   可通过 WxApiType 来获取，URL挂载参数需要自行拼接
     * @param mchId       商户Id
     * @param serialNo    商户 API 证书序列号
     * @param originalKey key.pem 证书字符串
     * @param body        接口请求参数
     * @param nonceStr    随机字符库
     * @param timestamp   时间戳
     * @param authType    认证类型
     * @return {@link String} 返回 v3 所需的 Authorization
     * @throws Exception 异常信息
     */
    public static String buildAuthorization(String method, String urlSuffix, String mchId,
                                            String serialNo, String originalKey, String body, String nonceStr,
                                            long timestamp, String authType) throws Exception {
        // 构建签名参数
        String buildSignMessage = buildSignMessage(method, urlSuffix, timestamp, nonceStr, body);
        String signature = createSign(buildSignMessage, originalKey);
        // 根据平台规则生成请求头 authorization
        return getAuthorization(mchId, serialNo, nonceStr, String.valueOf(timestamp), signature, authType);
    }

    /**
     * 私钥签名
     *
     * @param data       需要加密的数据
     * @param privateKey 私钥
     * @return 加密后的数据
     * @throws Exception 异常信息
     */
    public static String encryptByPrivateKey(String data, PrivateKey privateKey) throws Exception {
        Signature signature = Signature.getInstance("SHA256WithRSA");
        signature.initSign(privateKey);
        signature.update(data.getBytes(StandardCharsets.UTF_8));
        byte[] signed = signature.sign();
        return new String(Base64.getEncoder().encode(signed));
    }


    /**
     * 公钥验签
     */
    public static boolean verifySignByPublicKey(String content, String sign, String publicKey) {

        try {

            CertificateFactory cf = CertificateFactory.getInstance("X509");
            X509Certificate x509Cert = (X509Certificate) cf.generateCertificate(new ByteArrayInputStream(publicKey.getBytes(StandardCharsets.UTF_8)));

            BigInteger serialNumber = x509Cert.getSerialNumber();
            System.out.println(serialNumber);
            Signature signature = Signature.getInstance("SHA256WithRSA");

            signature.initVerify(x509Cert);

            signature.update(content.getBytes(StandardCharsets.UTF_8));

            return signature.verify(Base64.getDecoder().decode(sign));

        } catch (Exception e) {
            e.printStackTrace();
        }

        return false;

    }


    /**
     * v3 接口创建签名
     *
     * @param signMessage 待签名的参数
     * @param originalKey key.pem 证书字符串
     * @return 生成 v3 签名
     * @throws Exception 异常信息
     */
    public static String createSign(ArrayList<String> signMessage, String originalKey) throws Exception {
        return createSign(buildSignMessage(signMessage), originalKey);
    }

//    public static boolean checkSign(String signMessage, String originalKey,String sign){
//
//    }

    /**
     * v3 接口创建签名
     *
     * @param signMessage 待签名的参数
     * @param originalKey key.pem 证书字符串
     * @return 生成 v3 签名
     * @throws Exception 异常信息
     */
    public static String createSign(String signMessage, String originalKey) throws Exception {
        if (StringUtils.isEmpty(signMessage)) {
            return null;
        }
        // 获取商户私钥
        PrivateKey privateKey = getPrivateKeyByKeyContent(originalKey);
        // 生成签名
        return encryptByPrivateKey(signMessage, privateKey);
    }

    /**
     * 获取商户私钥
     *
     * @param originalKey 私钥文本内容
     * @return {@link PrivateKey} 商户私钥
     * @throws Exception 异常信息
     */
    public static PrivateKey getPrivateKeyByKeyContent(String originalKey) throws Exception {
        String privateKey = originalKey
                .replace("-----BEGIN PRIVATE KEY-----", "")
                .replace("-----END PRIVATE KEY-----", "")
                .replaceAll("\\s+", "");
        return loadPrivateKey(privateKey);
    }


    /**
     * 从字符串中加载私钥<br>
     * 加载时使用的是PKCS8EncodedKeySpec（PKCS#8编码的Key指令）。
     *
     * @param privateKeyStr 私钥
     * @return {@link PrivateKey}
     * @throws Exception 异常信息
     */
    public static PrivateKey loadPrivateKey(String privateKeyStr) throws Exception {
        try {
            byte[] buffer = Base64.getDecoder().decode(privateKeyStr);
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(buffer);
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            return keyFactory.generatePrivate(keySpec);
        } catch (NoSuchAlgorithmException e) {
            throw new Exception("无此算法");
        } catch (InvalidKeySpecException e) {
            throw new Exception("私钥非法");
        } catch (NullPointerException e) {
            throw new Exception("私钥数据为空");
        }
    }


    /**
     * <p>checkSign.</p>
     *
     * @param signMessage a {@link String} object
     * @param signature   a {@link String} object
     * @param certText    a {@link String} object
     * @return a boolean
     */
    public static boolean checkSign(String signMessage, String signature, String certText) {
        try {
            Signature sign = Signature.getInstance("SHA256withRSA");
            sign.initVerify(loadCert(certText));
            sign.update(signMessage.getBytes(StandardCharsets.UTF_8));
            return sign.verify(Base64.getDecoder().decode(signature));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    /**
     * 从字符串中加载证书<br>
     * 加载时使用的是PKCS8EncodedKeySpec（PKCS#8编码的Key指令）。
     *
     * @param cert 证书
     * @return {@link PrivateKey}
     * @throws Exception 异常信息
     */
    public static Certificate loadCert(String cert) throws Exception {
        CertificateFactory cf = CertificateFactory.getInstance("X509");
        return cf.generateCertificate(new ByteArrayInputStream(cert.getBytes(StandardCharsets.UTF_8)));
    }


    /**
     * 构造签名串
     *
     * @param method    {@link RequestMethod} GET,POST,PUT等
     * @param url       请求接口 /v3/certificates
     * @param timestamp 获取发起请求时的系统当前时间戳
     * @param nonceStr  随机字符串
     * @param body      请求报文主体
     * @return 待签名字符串
     */
    public static String buildSignMessage(String method, String url, long timestamp, String nonceStr, String body) {
        ArrayList<String> arrayList = new ArrayList<>();
        arrayList.add(method);
        arrayList.add(url);
        arrayList.add(String.valueOf(timestamp));
        arrayList.add(nonceStr);
        arrayList.add(body);
        return buildSignMessage(arrayList);
    }

    /**
     * 构造签名串
     *
     * @param timestamp 获取发起请求时的系统当前时间戳
     * @param nonceStr  随机字符串
     * @param body      请求报文主体
     * @return 待签名字符串
     */
    public static String buildSignMessage(long timestamp, String nonceStr, String body) {
        ArrayList<String> arrayList = new ArrayList<>();
        arrayList.add(String.valueOf(timestamp));
        arrayList.add(nonceStr);
        arrayList.add(body);
        return buildSignMessage(arrayList);
    }

    /**
     * 构造签名串
     *
     * @param signMessage 待签名的参数
     * @return 构造后带待签名串
     */
    public static String buildSignMessage(ArrayList<String> signMessage) {
        if (signMessage == null || signMessage.size() <= 0) {
            return null;
        }
        StringBuilder sbf = new StringBuilder();
        for (String str : signMessage) {
            sbf.append(str).append("\n");
        }
        return sbf.toString();
    }


    /**
     * 获取授权认证信息
     *
     * @param mchId     商户号
     * @param serialNo  商户API证书序列号
     * @param nonceStr  请求随机串
     * @param timestamp 时间戳
     * @param signature 签名值
     * @param authType  认证类型，目前为WECHATPAY2-SHA256-RSA2048
     * @return 请求头 Authorization
     */
    public static String getAuthorization(String mchId, String serialNo, String nonceStr, String timestamp, String signature, String authType) {
        Map<String, String> params = new LinkedHashMap<>(5);
        params.put("mchid", mchId);
        params.put("serial_no", serialNo);
        params.put("nonce_str", nonceStr);
        params.put("timestamp", timestamp);
        params.put("signature", signature);
        return authType.concat(" ").concat(createLinkString(params, ",", false, true));
    }

    /**
     * <p>createLinkString.</p>
     *
     * @param params  a {@link Map} object
     * @param connStr a {@link String} object
     * @param encode  a boolean
     * @param quotes  a boolean
     * @return a {@link String} object
     */
    public static String createLinkString(Map<String, String> params, String connStr, boolean encode, boolean quotes) {
        List<String> keys = new ArrayList<>(params.keySet());
        Collections.sort(keys);
        StringBuilder content = new StringBuilder();
        for (int i = 0; i < keys.size(); i++) {
            String key = keys.get(i);
            String value = params.get(key);
            // 拼接时，不包括最后一个&字符
            if (i == keys.size() - 1) {
                if (quotes) {
                    content.append(key).append("=").append('"').append(encode ? urlEncode(value) : value).append('"');
                } else {
                    content.append(key).append("=").append(encode ? urlEncode(value) : value);
                }
            } else {
                if (quotes) {
                    content.append(key).append("=").append('"').append(encode ? urlEncode(value) : value).append('"').append(connStr);
                } else {
                    content.append(key).append("=").append(encode ? urlEncode(value) : value).append(connStr);
                }
            }
        }
        return content.toString();
    }

    /**
     * URL 编码
     *
     * @param src 需要编码的字符串
     * @return 编码后的字符串
     */
    public static String urlEncode(String src) {
        try {
            return URLEncoder.encode(src, StandardCharsets.UTF_8.name()).replace("+", "%20");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
            return null;
        }
    }

}
