package io.gitee.open.nw.common.util;

import io.gitee.open.nw.common.base.BizException;
import io.gitee.open.nw.common.base.ResultEnum;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.lang.NonNull;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.*;
import java.util.function.Supplier;

/**
 * 对象工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class BeanUtil {


    private static final Logger log = LoggerFactory.getLogger(BeanUtil.class);


    public static <T> void treefy(List<T> dataList, String childrenFieldName, String idFieldName, String parentIdFieldName) {
        if (dataList.isEmpty()) {
            return;
        }
        Class<?> clazz = dataList.get(0).getClass();
        try {
            Field idField = clazz.getDeclaredField(idFieldName);
            Field parentIdField = clazz.getDeclaredField(parentIdFieldName);
            HashMap<Object, T> map = new HashMap<>();
            Iterator<T> iterator = dataList.iterator();
            List<T> children = new ArrayList<>();
            while (iterator.hasNext()) {
                T next = iterator.next();
                Object parentId = ClazzUtil.getValue(parentIdField, next);
                Object id = ClazzUtil.getValue(idField, next);
                if (parentId != null) {
                    iterator.remove();
                    children.add(next);
                }
                map.put(id, next);
            }
            Field childrenField = clazz.getDeclaredField(childrenFieldName);
            children.forEach(t -> {
                Object parentId = ClazzUtil.getValue(parentIdField, t);
                T parent = map.get(parentId);
                if (parent != null) {
                    List<T> value = ClazzUtil.getValue(childrenField, parent);
                    if (value==null){
                        value = new ArrayList<>();
                        ClazzUtil.setValue(childrenField, parent, value);
                    }
                    value.add(t);
                }
            });


        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        }

    }

    public static Map<String, Object> objectToMap(Object... objs) {
        try {
            Map<String, Object> map = new LinkedHashMap<>();
            for (Object obj : objs) {
                Class<?> clazz = obj.getClass();
                ReflectionUtils.doWithFields(clazz, (f) -> {
                    String name = f.getName();
                    if ("serialVersionUID".equals(name)) {
                        return;
                    }
                    ReflectionUtils.makeAccessible(f);
                    Object field = ReflectionUtils.getField(f, obj);
                    if (field != null) {
                        if (field instanceof Date date) {
                            map.put(name, DateFormatUtils.format(date, DateUtil.STR_DATETIME));
                        } else {
                            map.put(name, field);
                        }
                    }
                });
            }
            return map;
        } catch (Exception e) {
            log.error("对象转换Map失败", e);
            throw new BizException(ResultEnum.ERROR, "转换对象失败");
        }
    }

    public static boolean isLangPackage(Class<?> clazz) {
        return clazz.getPackage().getName().startsWith("java.lang");
    }

    public static Map<String, Object> objectToUnderlineMap(Object obj) {
        try {
            Map<String, Object> map = new LinkedHashMap<String, Object>();
            Class<?> clazz = obj.getClass();
            for (Field field : clazz.getDeclaredFields()) {
                field.setAccessible(true);
                String fieldName = field.getName();
                Object value = field.get(obj);
                if (value != null) {
                    map.put(StringUtil.camelToUnderline(fieldName), value);
                }
            }
            return map;
        } catch (Exception e) {
            log.error("对象转换Map失败", e);
        }
        return null;
    }

    public static <S, T> T convertTo(@NonNull S source, @NonNull Supplier<T> targetSupplier) {
        return convertTo(source, targetSupplier, null);
    }

    public static <S> void updateObject(S source, S target, String... ignore) {
        if (source == null) {
            return;
        }
        // 忽略更新的字段
        List<String> ignoreList = new ArrayList<>();
        ignoreList.add("createTime");
        ignoreList.add("updateTime");
        ignoreList.addAll(Arrays.asList(ignore));

        ReflectionUtils.doWithFields(source.getClass(), (f) -> {
            try {
                f.setAccessible(true);
                if (ReflectionUtils.getField(f, source) == null) {
                    ignoreList.add(f.getName());
                }
            } catch (Exception e) {
                throw new BizException(ResultEnum.PARAM_ERROR, e);
            }
        });
        BeanUtils.copyProperties(source, target, ignoreList.toArray(new String[]{}));
    }

    /**
     * 转换对象
     */
    public static <S, T> T convertTo(@NonNull S source, @NonNull Supplier<T> targetSupplier, ConvertCallBack<S, T> callBack) {
        T target = targetSupplier.get();
        BeanUtils.copyProperties(source, target);
        if (callBack != null) {
            callBack.callBack(source, target);
        }
        return target;
    }

    public static <S, T> List<T> convertListTo(@NonNull List<S> sources, @NonNull Supplier<T> targetSupplier) {
        return convertListTo(sources, targetSupplier, null);
    }

    /**
     * 转换对象
     */
    public static <S, T> List<T> convertListTo(@NonNull List<S> sources, @NonNull Supplier<T> targetSupplier, ConvertCallBack<S, T> callBack) {
        List<T> list = new ArrayList<>(sources.size());
        for (S source : sources) {
            T target = targetSupplier.get();
            BeanUtils.copyProperties(source, target);
            if (callBack != null) {
                callBack.callBack(source, target);
            }
            list.add(target);
        }
        return list;
    }

    /**
     * 回调接口
     */
    @FunctionalInterface
    public interface ConvertCallBack<S, T> {
        void callBack(S t, T s);
    }
}
