package io.gitee.open.nw.common.util;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import org.apache.poi.ss.usermodel.DateUtil;
import java.util.List;
import java.util.Map;

/**
 * excel poi 通用工具类
 *
 * @author CrazyZhang
 */
public class ExcelUtil {

    private final static String XLS = ".xls";
    private final static String XLSX = ".xlsx";


    /**
     * 读入excel文件，解析内容后返回结果
     */
    public static <T> List<T> readExcel(InputStream inputStream, String fileName, Integer startRowNum, RowsHandler<T> handler) throws IOException {
        return readExcel(inputStream, fileName, startRowNum, null, handler);
    }

    public static <T> List<T> readExcel(InputStream inputStream, String fileName, Integer startRowNum, Integer endRow, RowsHandler<T> handler) throws IOException{
        return readExcel(inputStream, fileName, startRowNum, endRow, null, handler);
    }

    /**
     * 读入excel文件，解析内容后返回结果
     */
    public static <T> List<T> readExcel(InputStream inputStream, String fileName, Integer startRowNum, Integer endRow,Integer columnNum, RowsHandler<T> handler) throws IOException {

        // 检查文件
        checkFile(fileName);
        // 获得Workbook工作薄对象
        Workbook workbook = getWorkBook(inputStream, fileName);
        // 创建返回对象，把每行中的值作为一个数组，所有行作为一个集合返回
        List<T> list = new ArrayList<>();
        if (workbook != null) {
            if (workbook.getNumberOfSheets() < 1 || workbook.getSheetAt(0).getLastRowNum() == 0) {
                throw new FileNotFoundException("文件不存在!");
            }

//          for (int sheetNum = 0; sheetNum < workbook.getNumberOfSheets(); sheetNum++) {// 循环导入各个Sheet
            // 获得当前sheet工作表
            Sheet sheet = workbook.getSheetAt(0);
            // 获得当前sheet的开始行
            int firstRowNum = sheet.getFirstRowNum();

            if (startRowNum != null) {
                if (firstRowNum < startRowNum - 1) {
                    firstRowNum = startRowNum - 1;
                }
            }

            int lastRowNum = 0;
            if (endRow == null || endRow == 0) {
                lastRowNum = sheet.getLastRowNum();
            } else if (endRow > 0 && endRow > firstRowNum) {
                lastRowNum = sheet.getFirstRowNum() + endRow;
            } else if (endRow < 0) {
                // 获得当前sheet的结束行
                lastRowNum = sheet.getLastRowNum() + endRow;
            } else {
                lastRowNum = sheet.getLastRowNum();
            }

            if(columnNum==null){
                // 根据行头确定每一行的列数，这里规定了行头的列数 = 数据的列数
                columnNum = (int) sheet.getRow(firstRowNum).getLastCellNum();
            }
            // 循环除了第一行的所有行
            for (int rowNum = firstRowNum; rowNum <= lastRowNum; rowNum++) {
                // 获得当前行
                Row row = sheet.getRow(rowNum);
                // 获得当前行的开始列
                int firstCellNum = row.getFirstCellNum();
                Object[] cellDataArr = new String[columnNum];
                // 循环当前行
                for (int cellNum = firstCellNum; cellNum < columnNum; cellNum++) {
                    Cell cell = row.getCell(cellNum);
                    cellDataArr[cellNum] = getCellValue(cell);
                }

                // 结果添加到list
                T value = handler.execute(rowNum + 1, cellDataArr);
                if (value != null) {
                    list.add(value);
                }
            }
        }
        return list;
    }

    public static void writeExcel(OutputStream out, String filename, Map<String, String> header, List<Map<String, Object>> dataset) throws IOException {

        writeExcel(out, filename, header, null, Collections.singletonList(dataset));
    }

    /**
     * 批量导出多个sheet
     */
    public static void writeExcel(OutputStream out, String filename, Map<String, String> header, List<String> sheetNames, List<List<Map<String, Object>>> sheetListData) throws IOException {

        // 获得Workbook工作薄对象
        Workbook workbook = getWorkBook(filename);
        CellStyle defaultTitleStyle = getDefaultTitleStyle(workbook);

        for (int sheetIndex = 0; sheetIndex < sheetListData.size(); sheetIndex++) {
            // 需要导出的数据
            List<Map<String, Object>> dataset = sheetListData.get(sheetIndex);
            Sheet sheet = workbook.createSheet();
            if (sheetNames != null) {
                workbook.setSheetName(sheetIndex, sheetNames.get(sheetIndex));
            }

            // 设置标题
            Row headRow = sheet.createRow(0);
            List<String> values = new ArrayList<String>(header.values());
            for (int i = 0; i < values.size(); i++) {
                Cell cell = headRow.createCell(i);
                cell.setCellValue(values.get(i));
                cell.setCellStyle(defaultTitleStyle);
            }

            CellStyle defaultDataStyle = getDefaultDataStyle(workbook);
            List<String> headKeys = new ArrayList<String>(header.keySet());

            for (int i = 0; i < dataset.size(); i++) {
                Map<String, Object> data = dataset.get(i);
                Row row = sheet.createRow(i + 1);
                for (int j = 0; j < headKeys.size(); j++) {

                    Cell cell = row.createCell(j);
                    cell.setCellStyle(defaultDataStyle);
                    String headKey = headKeys.get(j);
                    Object value = data.get(headKey);
                    String textVal = null;
                    if (value != null) {
                        textVal = value.toString();
                    }
                    cell.setCellValue(textVal);
                }
            }
        }
        workbook.write(out);
    }


    /**
     * 设置标题样式
     */
    private static CellStyle getDefaultTitleStyle(Workbook workbook) {
        CellStyle cellStyle = workbook.createCellStyle();
        cellStyle.setFillForegroundColor(IndexedColors.AQUA.getIndex());
        cellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        cellStyle.setAlignment(HorizontalAlignment.CENTER);// 水平
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);// 垂直
        Font font = workbook.createFont();
        font.setFontHeightInPoints((short) 11);
        font.setBold(true);
        cellStyle.setFont(font);
        return cellStyle;
    }


    /**
     * 设置数据样式
     */
    private static CellStyle getDefaultDataStyle(Workbook workbook) {
        CellStyle cellStyle = workbook.createCellStyle();
        Font font = workbook.createFont();
        font.setFontHeightInPoints((short) 9);
        cellStyle.setFont(font);
        // 水平居中
        cellStyle.setAlignment(HorizontalAlignment.CENTER);
        return cellStyle;
    }


    private static void checkFile(String fileName) throws IOException {
        // 判断文件是否是excel文件
        if (!fileName.endsWith(XLS) && !fileName.endsWith(XLSX)) {
            throw new IOException(fileName + "不是excel文件");
        }
    }

    /**
     * 获取工作区间
     */
    private static Workbook getWorkBook(InputStream is, String fileName) {
        // 创建Workbook工作薄对象
        Workbook workbook = null;
        try {
            // 根据文件后缀名不同(xls和xlsx)获得不同的Workbook实现类对象
            if (fileName.endsWith(XLS)) {
                // 2003
                workbook = new HSSFWorkbook(is);
            } else if (fileName.endsWith(XLSX)) {
                // 2007
                workbook = new XSSFWorkbook(is);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return workbook;
    }


    private static Workbook getWorkBook(String fileName) throws IOException {
        // 创建Workbook工作薄对象
        // 根据文件后缀名不同(xls和xlsx)获得不同的Workbook实现类对象
        if (fileName.endsWith(XLS)) {
            // 2003
            return new HSSFWorkbook();
        } else if (fileName.endsWith(XLSX)) {
            // 2007
            return new XSSFWorkbook();
        } else {
            throw new IOException("文件名后缀错误");
        }
    }


    /**
     * 获取各个单元格的内容
     */
    private static Object getCellValue(Cell cell) {
        if (cell == null) {
            return "";
        }
        // 判断数据的类型
        if (cell.getCellType() == CellType.NUMERIC) {
            // 数值
            if (DateUtil.isCellDateFormatted(cell)) {
                return cell.getDateCellValue();
            }
        }
        cell.setCellValue(cell.getStringCellValue());
        return cell.getStringCellValue();
    }

    /**
     * 每读取一行，就调用execute方法
     */
    public interface RowsHandler<T> {

        /**
         * 可以再此方法校验数据，校验通过后封装 T，再返回，读入完后，再批量insert/update
         *
         * @param lineNum 行号
         * @param rows    读取的当前这一行的数据
         * @return 自定义封装成 T,返回null则不记录到list
         */
        T execute(int lineNum, Object[] rows);

    }


}
