package io.gitee.open.nw.common.util;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.Inet4Address;
import java.net.NetworkInterface;
import java.nio.charset.StandardCharsets;

/**
 * 雪花id生成器
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class SnowFlakeWorker {


    private static final Logger logger = LoggerFactory.getLogger(SnowFlakeWorker.class);

    private volatile static SnowFlakeWorker snowFlakeWorkerInstance;

    // 1位标识部分    -      41位时间戳部分        -         10位节点部分     12位序列号部分
    /* 0 - 0000000000 0000000000 0000000000 0000000000 0 - 00000 - 00000 - 000000000000 */
    /**
     * 起始的时间戳
     */
    private final static long START_STMP = 1668096000000L;


    //---------------每一部分占用的位数---------------
    /**
     * 序列号占用的位数
     */
    private final static long SEQUENCE_BIT = 12;

    /**
     * 机器标识占用的位数
     */
    private final static long WORK_BIT = 10;

    /**
     * WORK_NUM最大值 1023
     */
    private final static long MAX_WORK_NUM = ~(-1L << WORK_BIT);
    /**
     * SEQUENCE最大值 4095
     */
    private final static long MAX_SEQUENCE = ~(-1L << SEQUENCE_BIT);

    /**
     * 每一部分向左的位移
     */
    private final static long WORK_LEFT = SEQUENCE_BIT;
    private final static long TIMESTMP_LEFT = WORK_LEFT + WORK_BIT;

    private final long workId;
    /**
     * 序列号
     */
    private long sequence = 0L;
    /**
     * 上一次时间戳
     */
    private long lastStmp = -1L;

    /**
     * 步长, 1024
     */
    private final static long STEP_SIZE = 2 << 9;
    /**
     * 基础序列号, 每发生一次时钟回拨即改变, basicSequence += stepSize
     */
    private long basicSequence = 0L;

    private SnowFlakeWorker(long workId) {
        logger.info("雪花算法机器id:{}", workId);
        if (workId > MAX_WORK_NUM || workId < 0) {
            throw new IllegalArgumentException("datacenterId can't be greater than MAX_DATACENTER_NUM or less than 0");
        }
        this.workId = workId;
    }

    /**
     * <p>getInstance.</p>
     *
     * @return a {@link SnowFlakeWorker} object
     */
    public static SnowFlakeWorker getInstance() {
        if (snowFlakeWorkerInstance == null) {
            synchronized (SnowFlakeWorker.class) {
                if (snowFlakeWorkerInstance == null) {
                    snowFlakeWorkerInstance = new SnowFlakeWorker(getWorkerId());
                }
            }
        }
        return snowFlakeWorkerInstance;
    }

    /**
     * 获取本地mac地址
     * 注意：物理地址是48位，别和ipv6搞错了
     *
     * @return 本地mac地址
     */
    private static String getLocalMac() {
        try {
            //获取网卡，获取地址
            byte[] mac = NetworkInterface.getByInetAddress(Inet4Address.getLocalHost()).getHardwareAddress();
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < mac.length; i++) {
                if (i != 0) {
                    sb.append("-");
                }
                //字节转换为整数
                int temp = mac[i] & 0xff;
                String str = Integer.toHexString(temp);
                if (str.length() == 1) {
                    sb.append("0").append(str);
                } else {
                    sb.append(str);
                }
            }
            return sb.toString();
        } catch (Exception exception) {
            logger.info("获取网卡mac地址错误");
        }
        return "";
    }

    /**
     * <p>getWorkerId.</p>
     *
     * @return a long
     */
    public static long getWorkerId() {
        String path = System.getProperty("user.dir");
        if (StringUtils.isBlank(path)){
            if (isWindows()){
                path = "D://";
            } else {
                path = "";
            }
        }
        String filePath = path + File.separator + "SnowFlakeWorkerId.txt";
        File file = new File(filePath);
        if (file.exists()){
            String workerId;
            try {
                workerId = FileUtils.readFileToString(file, StandardCharsets.UTF_8);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            return Long.parseLong(workerId);
        }
        try {
            file.createNewFile();

            // 获取pid
            RuntimeMXBean bean = ManagementFactory.getRuntimeMXBean();
            String jvmName = bean.getName();
            long pid = Long.parseLong(jvmName.split("@")[0]);
            String localMac = getLocalMac();
            long workId = (Math.abs(localMac.hashCode()) + pid) % (MAX_WORK_NUM + 1);
            FileUtils.writeStringToFile(file,workId+"",StandardCharsets.UTF_8);
            return workId;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }

    /**
     * <p>isWindows.</p>
     *
     * @return a boolean
     */
    public static boolean isWindows() {
        return (System.getProperty("os.name").toLowerCase().contains("win"));
    }


    /**
     * 产生下一个ID
     *
     * @return a long
     */
    public synchronized long nextId() {
        long currStmp = getNewstmp();
        if (currStmp < lastStmp) {
            return handleClockBackwards(currStmp);
        }

        if (currStmp == lastStmp) {
            // 相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            // 同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currStmp = getNextMill();
            }
        } else {
            // 不同毫秒内，序列号置为 basicSequence
            sequence = basicSequence;
        }
        lastStmp = currStmp;
        return (currStmp - START_STMP) << TIMESTMP_LEFT  // 时间戳部分
                | workId << WORK_LEFT                    // 节点部分
                | sequence;                              // 序列号部分
    }

    /**
     * 处理时钟回拨
     */
    private long handleClockBackwards(long currStmp) {
        basicSequence += STEP_SIZE;
        if (basicSequence == MAX_SEQUENCE + 1) {
            basicSequence = 0;
            currStmp = getNextMill();
        }
        sequence = basicSequence;

        lastStmp = currStmp;
        return (currStmp - START_STMP) << TIMESTMP_LEFT  // 时间戳部分
                | workId << WORK_LEFT                    // 节点部分
                | sequence;                              // 序列号部分
    }

    private long getNextMill() {
        long mill = getNewstmp();
        while (mill <= lastStmp) {
            mill = getNewstmp();
        }
        return mill;
    }

    private long getNewstmp() {
        return System.currentTimeMillis();
    }


}
