
package io.gitee.open.nw.common.util.encrypt;

import org.bouncycastle.jcajce.provider.asymmetric.rsa.CipherSpi;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

/**
 * rsa 加密工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class RSAUtil {

    /**
     * 加密算法RSA
     */
    public static final String KEY_ALGORITHM = "RSA";

    /**
     * 签名算法
     */
    public static final String SIGNATURE_ALGORITHM = "MD5withRSA";

    /**
     * 获取公钥的key
     */
    private static final String PUBLIC_KEY = "RSAPublicKey";

    /**
     * 获取私钥的key
     */
    private static final String PRIVATE_KEY = "RSAPrivateKey";

    /**
     * RSA最大加密明文大小
     */
    private static final int MAX_ENCRYPT_BLOCK = 240;

    /**
     * RSA最大解密密文大小
     */
    private static final int MAX_DECRYPT_BLOCK = 256;

    /**
     * 生成密钥对(公钥和私钥)
     */
    public static Map<String, Object> genKeyPair() throws Exception {
        KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        keyPairGen.initialize(1024);
        KeyPair keyPair = keyPairGen.generateKeyPair();
        RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
        RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
        Map<String, Object> keyMap = new HashMap<String, Object>(2);
        keyMap.put(PUBLIC_KEY, publicKey);
        keyMap.put(PRIVATE_KEY, privateKey);
        return keyMap;
    }

    /**
     * 用私钥对信息生成数字签名
     */
    public static byte[] sign(byte[] data, byte[] privateKey) throws Exception {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(privateKey);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        PrivateKey privateK = keyFactory.generatePrivate(pkcs8KeySpec);
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);

        signature.initSign(privateK);
        signature.update(data);
        return Base64.getEncoder().encode(signature.sign());
    }

    /**
     * 校验数字签名
     */
    public static boolean verify(byte[] data, byte[] publicKey, byte[] sign)
            throws Exception {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKey);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        PublicKey publicK = keyFactory.generatePublic(keySpec);
        Signature signature = Signature.getInstance(SIGNATURE_ALGORITHM);
        signature.initVerify(publicK);
        signature.update(data);
        return signature.verify(Base64Util.decode(sign));
    }

    /**
     * 私钥解密
     */
    public static byte[] decryptByPrivateKey(byte[] encryptedData, byte[] privateKey)
            throws Exception {
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(privateKey);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key privateK = keyFactory.generatePrivate(pkcs8KeySpec);
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.DECRYPT_MODE, privateK);
        return getByteArray(encryptedData, cipher, MAX_DECRYPT_BLOCK);
    }

    private static byte[] getByteArray(byte[] data, Cipher cipher, int maxBlockSize) throws IllegalBlockSizeException, BadPaddingException, IOException {
        int inputLen = data.length;
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        int offSet = 0;
        byte[] cache;
        int i = 0;
        // 对数据分段解密
        while (inputLen - offSet > 0) {
            if (inputLen - offSet > maxBlockSize) {
                cache = cipher.doFinal(data, offSet, maxBlockSize);
            } else {
                cache = cipher.doFinal(data, offSet, inputLen - offSet);
            }
            out.write(cache, 0, cache.length);
            i++;
            offSet = i * maxBlockSize;
        }
        byte[] endeData = out.toByteArray();
        out.close();
        return endeData;
    }

    /**
     * 公钥加密
     */
    public static byte[] encryptByPublicKey(byte[] data, byte[] publicKey)
            throws Exception {
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(publicKey);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        Key publicK = keyFactory.generatePublic(x509KeySpec);
        // 对数据加密
        Cipher cipher = Cipher.getInstance(keyFactory.getAlgorithm());
        cipher.init(Cipher.ENCRYPT_MODE, publicK);
        return getByteArray(data, cipher, MAX_ENCRYPT_BLOCK);
    }

    /**
     * 获取私钥
     */
    public static byte[] getPrivateKey(Map<String, Object> keyMap)
            throws Exception {
        Key key = (Key) keyMap.get(PRIVATE_KEY);
        return Base64Util.encode(key.getEncoded());
    }

    /**
     * 获取公钥
     */
    public static byte[] getPublicKey(Map<String, Object> keyMap)
            throws Exception {
        Key key = (Key) keyMap.get(PUBLIC_KEY);
        return Base64Util.encode(key.getEncoded());
    }

    public static void main(String[] args) throws Exception {
//        Map<String, Object> keyPair = genKeyPair();
//        System.out.println(new String(getPrivateKey(keyPair), StandardCharsets.UTF_8));
//        System.out.println(new String(getPublicKey(keyPair), StandardCharsets.UTF_8));

        String pk="MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQCWf63vC0DcDftnzGIfWZjkHWEOtmY45ffMHmveYj4giHdXiWGgBTQAQ5HklxDztSMVh6I/QlnX422I+gviqU6cJ1BFR4TlwRIWvc8CD40Ehvu+UjTh+yS7K8hEjGz3+1aZFLRBMR3Rv6CurldzAHoSLgrjH8ccsKKMgE0+Wkb4EApITJQgPkkBEMsTcLpOsheYYq/oyYO520E29OOZ0YZx7t5QBfHVOIhxA1Htz6ylk7ZcYUu07fsDAQSZDhprkBxA7dOwrmRs05jMlODrmmeaguiP+r4smEmE6lcvFLrRApvZShmICcH0Lgi3Cl+PoUP00H18IW3plzetpcnQV1bFAgMBAAECggEACcFM7Om10o21dgzxb0bHekHNQL7h8V42qaxvIYMMs98knS/yfATaUgoPMtn351Qeag7YnXEXkb7UIsTScGa5CbYipdr06k+RdJfpCyEm6tw1BSke9GbrApbb7FWN2qUX/+QvquocHLRnHet/D51ATF1fp0kwBtQMNT87wobngrytRKYLyKf4pup3hDT7DkR1rdzeCbkaeRs+3lQB7gKQy5b5Q1PJR0rn75GdkpxAuBQBXe+ui5KgS064AWTCFALtDj3kUvZzBxx4p90kjFroJCMAq8KM1oIA0F7pmyYqfQ7PtdrzK+qGDWUd8CZCC/tp0QEgO3hsMvrTwe3t5L9YAQKBgQD4qqMsw3cky2kRl+Z2/OOzy83g47CHSSUbqZgl3FGqV89DvF3D0DY9fYTolX9cWwC3Q7PhxV72mDu8NTys5W8MKofLd+3hG3UK9zsUn0iZl2n6CzB4Y8Vd4jec18rZUNSbsRTujC/yUbOXo/k+8zPpHAA6rJlg7BpqkSq4sftbhQKBgQCa7+cWcj8bmUOKUNftTvFYkhe1EiqSPjo/AldsJGZhRn3OBhg70LBhdg1JwNJIUQn9iHaqV5+odW+/8jw/MfDJoRGCJNGCBL6CAGQjeDltpdxg2mlVJfDaIvSoYPHMF/c8I7VY2j34nm4wJ4tzZhoVKC8pdsisXQD5qSesMkrSQQKBgQDzI2WJT1Yf1ARX/36IuZyrIIudaQQiQ5LVCW6LFnGDbXhrcrL704o1dZWPFR4C37hDeR9ZkfLYiuFdlXBOXcfGzY8Zk5o0kPRQVIygDV6fTz2AD5DFJkjFE3vtD9J3ymEYGn2tAcWf2eMQUnRS6NnJBGf09u7BphnuOSLclKqBQQKBgDxobOEDijdoejzjaEzOCqtNypB+lLqQKIWIlC1Fw+UJZuN83NX1d8+IB4ZJ97LaZdrD6gBgSY+Uug1MlNDO1+LFOd5UHJtk3iIxpNlvdnSZxmK8xXS9Zjqmhl1f7uZ0MaDxCzOpY6AbpTqBMI29sU2ubTTtazBI0qiEm0M4GJQBAoGBANXRErRFKT7O1Kyo9oAxQfIrXqmVMHiu0wTRKyyRBtNKVhyZITEQLAlTQ7Y/XYIKSYwHPV9iwCAVID4sX3PEHtPj2Znq+q4/vJ40usfVBCI6+YeNNhVzBT2yrfBV/35VutYjlMZI/Sj88BaYlHDFBTyrYRkaMd+ko69S4VAyX/SQ";
        String data = "aVPYLbg01KDuyi/NyM2fGasHik9/n9AMrZ7nCSxSlpuJhaOVTNd8vq8vLKn+nTxyHaDJkeynCvGdbnGOLzAlCnA8bHeKWtVQ81X6UWfeeejNj8XqHgvqFs34OeAvK6lxQ8L7QgxT+jGGnwTMDaMFpeIpwgcQHu2DeAoSG7CD3DXUFQPUTW9FUiGwiOot2Jw52D9hKEy0q0jRFDWVD4Ha3Hl5Ob6LmJBAHYvoV9zDFoXh//nCsfZloLtn76AGWvy5rO5o8Jra7PjIi1QPIX9zd9Ih1+Z7Y94Uagh4f9T53iCMMepu0nMIVZ880uG+P94IFa3JE7XtFIcyhRnNrpwtRw==";
        byte[] decrypt = decryptByPrivateKey(data.getBytes(), pk.getBytes());
        System.out.println(new String(decrypt));




    }

}
