package io.gitee.open.nw.common.util.encrypt;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.annotation.JSONField;
import io.gitee.open.nw.common.base.BizException;
import io.gitee.open.nw.common.base.ResultEnum;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;

/**
 * 综合加密工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class SignEncryptUtil {
    private final static Logger log = LoggerFactory.getLogger(SignEncryptUtil.class);


    /**
     * 签名参数生成
     */
    public static SignEncryptData paramSign(byte[] privatekey, byte[] publickey, byte[] param, byte[] caller) throws Exception {
        //1. 生成AES 16位明文密码
        String encodeRules = getRandomString(16);

        //私钥加签、公钥验签（我方用我方自己的私钥加签，对方用我方公钥验签）
        //我方用自己的私钥对消息加签，形成签名，将加签的消息传递给对方
        //对方收到消息后，再获取我方的公钥进行验签，如果验签出来的内容与消息本身一致，证明消息是我方回复的。
        //2.1 用生成的AES密码加密数据内容
        byte[] enContent = AesUtil.encryptAesECB(param, encodeRules.getBytes(StandardCharsets.UTF_8));

        enContent = Base64Util.encode(enContent);
        long timeMillis = System.currentTimeMillis();

        assert enContent != null;
        String signStr = new String(enContent, StandardCharsets.UTF_8) + "&" + new String(caller) + "&" + timeMillis;

        //2.2 使用我方私钥生成签名（签名=我方私钥+AES密码）
        byte[] sign = RSAUtil.sign(signStr.getBytes(StandardCharsets.UTF_8), privatekey);

        //公钥加密、私钥解密（我方用对方的公钥加密，对方用对方自己的私钥解密）
        //对方传递自己的公钥给我方，我方用对方的公钥对消息进行加密。
        //对方接收到我方加密的信息，利用他们自己的私钥对信息进行解密。
        //3. 使用对方公钥对16位明文密码进行RSA加密
        byte[] enKey = RSAUtil.encryptByPublicKey(encodeRules.getBytes(StandardCharsets.UTF_8), publickey);
        enKey = Base64Util.encode(enKey);
        SignEncryptData signEncryptDataObj = new SignEncryptData();
        signEncryptDataObj.setEncKeySrc(encodeRules);
        signEncryptDataObj.setEncData(new String(enContent, StandardCharsets.UTF_8));
        signEncryptDataObj.setTimestamp(timeMillis);
        signEncryptDataObj.setCaller(new String(caller));
        signEncryptDataObj.setSign(new String(sign, StandardCharsets.UTF_8));
        signEncryptDataObj.setEncKey(new String(enKey, StandardCharsets.UTF_8));
        return signEncryptDataObj;
    }

    public static byte[] encrypt(byte[] data, byte[] signRuleSrc) throws Exception {
        return Base64Util.encode(AesUtil.encryptAesECB(data, signRuleSrc));
    }

    /**
     * <p>decryptResponse.</p>
     *
     * @param data        an array of {@link byte} objects
     * @param signRuleSrc an array of {@link byte} objects
     * @return an array of {@link byte} objects
     * @throws Exception if any.
     */
    public static byte[] decrypt(byte[] data, byte[] signRuleSrc) throws Exception {
        return AesUtil.decryptAesECB(Base64Util.decode(data), signRuleSrc);
    }


    /**
     * 签名校验
     */
    public static boolean checkSign(SignEncryptData signEncryptData, byte[] publickey, Long timeMillis) throws Exception {
        if (timeMillis == null) {
            timeMillis = 300000L;
        }
        long now = System.currentTimeMillis();
        //5分钟之内请求
        if ((now - signEncryptData.getTimestamp()) > timeMillis) {
            throw new BizException(ResultEnum.PARAM_ERROR, "请求不合法，时间戳大于五分钟");
        }
        return RSAUtil.verify((signEncryptData.getEncData() + "&" + signEncryptData.getCaller() + "&" + signEncryptData.getTimestamp()).getBytes(StandardCharsets.UTF_8), publickey, signEncryptData.getSign().getBytes(StandardCharsets.UTF_8));
    }

    /**
     * 验签 + 解密
     */
    public static byte[] checkSignDecrypt(SignEncryptData signEncryptData, byte[] privateKey, byte[] publicKey) throws Exception {
        //1. 验证签名
        boolean signd = checkSign(signEncryptData, publicKey, null);
        if (!signd) {
            throw new BizException(ResultEnum.PARAM_ERROR, "验签失败");
        }
        log.info(">>> 验签" + (signd ? "成功" : "失败"));

        //2.解密数据
        return decryptData(signEncryptData, privateKey);
    }

    /**
     * 验签 + 解密
     */
    public static byte[] checkSignDecrypt(SignEncryptData signEncryptData, byte[] privateKey, byte[] publicKey, Long timeMillis) throws Exception {
        //1. 验证签名
        boolean signd = checkSign(signEncryptData, publicKey, timeMillis);
        if (!signd) {
            throw new BizException(ResultEnum.PARAM_ERROR, "验签失败");
        }
        log.info(">>> 验签" + (signd ? "成功" : "失败"));

        //2.解密数据
        return decryptData(signEncryptData, privateKey);
    }

    /**
     * 解密数据
     */
    public static byte[] decryptData(SignEncryptData signEncryptData, byte[] privateKey) throws Exception {

        byte[] encRule = Base64Util.decode(signEncryptData.getEncKey().getBytes(StandardCharsets.UTF_8));

        //解密RSA算法加密过的AES密码
        byte[] bytesKey = RSAUtil.decryptByPrivateKey(encRule, privateKey);

        signEncryptData.setEncKeySrc(new String(bytesKey, StandardCharsets.UTF_8));

        byte[] encData = Base64Util.decode(signEncryptData.getEncData().getBytes(StandardCharsets.UTF_8));
        return AesUtil.decryptAesECB(encData, bytesKey);
    }

    public static String getRandomString(int length) { //length表示生成字符串的长度
        String base = "abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        Random random = new Random();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < length; i++) {
            int number = random.nextInt(base.length());
            sb.append(base.charAt(number));
        }
        return sb.toString();
    }

    public static void main(String[] args) throws Exception {
        //A方私钥
        String privatekey = "MIICdQIBADANBgkqhkiG9w0BAQEFAASCAl8wggJbAgEAAoGBAIAy+fL99zuhASUNu73wL5j9fumLjpAEumt709Lpuz/28eSX0O9OR+vm8TKDm1SfV2rIuWrBbnv1RUJ/QyRmhqAz6cV3wv6CmVPIg/nrguTShA1pYE0ZxyQM5sqAz2oCyntyT6xtYleo5lCiEcUmQLPW2g0JzXqYds7IgNYlcahpAgMBAAECgYAU5bpexM1XWAByxNFUveFUZAYbFdewXETzHmVCHT2byKpXo/O3/p+0sZmhrVhVq/+49W1cHJng9PNMSqhsN5Y8LP1wvFpSXVMw6nxAwMWykohsss2wAKtZGUlF6K8dD07NfDPHmvDACcVbya5jPfs6Oz6cN1bT9ygIU9BGNzOQXQJBAJYPonFnfBveOj/1u6z7N2xeHfSALqYPJF+r7C10v5oUaIwirsd+TA0JNUbkMQHD5DMea2jxCOjLakKcy+71dS8CQQDatEHk/8jcpBbVVHxhaOZzO3Lp7U2s1r/MnOvwEtqQr3DiElkliOnVAbWbZT9Nt5aGEfXJ6MEBUAJNFVTrRQXnAkBdS2EaElWV2ebrJvLEi/ccN7N19E+MTHO3y2JtrF0HBVMudM2nF71eiXAHIA3b5Bc3NN1ghCKe4QCyY1tiuFZfAkBMnd+DyPPmD/oLzCgyu9wXvFjaiI6woR7aIJMrLH+zNjL2j/T5CEjanVmw0bxmjo1F9J7FmCudORlFRdbVAP47AkBMTgCehCnorAtqJf9GZm21/7CpV6MRscUVpC1lOe1sYfCIKa7LYo6m+UgTsh1AhW1s0m//HQBcX9x6x8RpQvsr";
        //B方公钥
        String publickey = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCAMvny/fc7oQElDbu98C+Y/X7pi46QBLpre9PS6bs/9vHkl9DvTkfr5vEyg5tUn1dqyLlqwW579UVCf0MkZoagM+nFd8L+gplTyIP564Lk0oQNaWBNGcckDObKgM9qAsp7ck+sbWJXqOZQohHFJkCz1toNCc16mHbOyIDWJXGoaQIDAQAB";
        //发送方传递的信息
        String param = "{}";
        //得到加密的数据
        SignEncryptData signEncryptData = paramSign(privatekey.getBytes(StandardCharsets.UTF_8), publickey.getBytes(StandardCharsets.UTF_8), param.getBytes(StandardCharsets.UTF_8),"".getBytes());
        System.out.println("加密数据：" + JSON.toJSONString(signEncryptData));
        //B方私钥
        String c_private_key = "MIICdQIBADANBgkqhkiG9w0BAQEFAASCAl8wggJbAgEAAoGBAIAy+fL99zuhASUNu73wL5j9fumLjpAEumt709Lpuz/28eSX0O9OR+vm8TKDm1SfV2rIuWrBbnv1RUJ/QyRmhqAz6cV3wv6CmVPIg/nrguTShA1pYE0ZxyQM5sqAz2oCyntyT6xtYleo5lCiEcUmQLPW2g0JzXqYds7IgNYlcahpAgMBAAECgYAU5bpexM1XWAByxNFUveFUZAYbFdewXETzHmVCHT2byKpXo/O3/p+0sZmhrVhVq/+49W1cHJng9PNMSqhsN5Y8LP1wvFpSXVMw6nxAwMWykohsss2wAKtZGUlF6K8dD07NfDPHmvDACcVbya5jPfs6Oz6cN1bT9ygIU9BGNzOQXQJBAJYPonFnfBveOj/1u6z7N2xeHfSALqYPJF+r7C10v5oUaIwirsd+TA0JNUbkMQHD5DMea2jxCOjLakKcy+71dS8CQQDatEHk/8jcpBbVVHxhaOZzO3Lp7U2s1r/MnOvwEtqQr3DiElkliOnVAbWbZT9Nt5aGEfXJ6MEBUAJNFVTrRQXnAkBdS2EaElWV2ebrJvLEi/ccN7N19E+MTHO3y2JtrF0HBVMudM2nF71eiXAHIA3b5Bc3NN1ghCKe4QCyY1tiuFZfAkBMnd+DyPPmD/oLzCgyu9wXvFjaiI6woR7aIJMrLH+zNjL2j/T5CEjanVmw0bxmjo1F9J7FmCudORlFRdbVAP47AkBMTgCehCnorAtqJf9GZm21/7CpV6MRscUVpC1lOe1sYfCIKa7LYo6m+UgTsh1AhW1s0m//HQBcX9x6x8RpQvsr";
        //A方公钥
        String s_public_key = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQCAMvny/fc7oQElDbu98C+Y/X7pi46QBLpre9PS6bs/9vHkl9DvTkfr5vEyg5tUn1dqyLlqwW579UVCf0MkZoagM+nFd8L+gplTyIP564Lk0oQNaWBNGcckDObKgM9qAsp7ck+sbWJXqOZQohHFJkCz1toNCc16mHbOyIDWJXGoaQIDAQAB";
        byte[] bytes = checkSignDecrypt(signEncryptData, c_private_key.getBytes(StandardCharsets.UTF_8), s_public_key.getBytes(StandardCharsets.UTF_8));
        System.out.println("解密数据：" + new String(bytes, StandardCharsets.UTF_8));
    }

    /**
     * 使用 Map按key进行排序
     */
    public static Map<String, String> sortMapByKey(Map<String, String> map) {
        if (map == null || map.isEmpty()) {
            return null;
        }
        Map<String, String> sortMap = new TreeMap<>(String::compareTo);
        sortMap.putAll(map);
        return sortMap;
    }

    public static String getSortedQueryString(Map<String, String> params) {
        Map<String, String> resultMap = sortMapByKey(params);    //按Key进行排序
        StringBuilder buffer = new StringBuilder();
        Set<Map.Entry<String, String>> entries = params.entrySet();
        for (Map.Entry<String, String> entry : entries) {
            buffer.append(entry.getKey()).append("=").append(entry.getValue()).append("&");
        }
        return buffer.substring(0, buffer.length() - 1);
    }


    public static class SignEncryptData {

        private String caller;

        private String encData;

        private Long timestamp;

        private String encKey;

        @JSONField(serialize = false)
        private String encKeySrc;

        private String sign;

        public String getEncData() {
            return encData;
        }

        public void setEncData(String encData) {
            this.encData = encData;
        }

        public Long getTimestamp() {
            return timestamp;
        }

        public void setTimestamp(Long timestamp) {
            this.timestamp = timestamp;
        }

        public String getEncKey() {
            return encKey;
        }

        public void setEncKey(String encKey) {
            this.encKey = encKey;
        }

        public String getSign() {
            return sign;
        }

        public void setSign(String sign) {
            this.sign = sign;
        }

        public String getEncKeySrc() {
            return encKeySrc;
        }

        public void setEncKeySrc(String encKeySrc) {
            this.encKeySrc = encKeySrc;
        }

        public String getCaller() {
            return caller;
        }

        public void setCaller(String caller) {
            this.caller = caller;
        }
    }
}
