package io.gitee.open.nw.common.util.encrypt;

import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.Security;

/**
 * 国密加密工具
 *
 * @author CrazyZhang
 * @since 2024/3/22 13:34
 */
public class Sm4Util {

    public static final String ALGORITHM_NAME = "SM4";
    public static final String ALGORITHM_NAME_ECB_PADDING = "SM4/ECB/PKCS7Padding";

    private static final Logger logger = LoggerFactory.getLogger(Sm4Util.class);

    public Sm4Util() {
    }

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 加密
     *
     * @param hexKey  明文密钥
     * @param content 内容
     * @return 加密数据
     */
    public static String encrypt(String hexKey, String content) {
        String encryptContent = StringUtils.EMPTY;
        try {
            byte[] keyData = hexKey.getBytes(StandardCharsets.UTF_8);
            byte[] contentData = content.getBytes(StandardCharsets.UTF_8);
            byte[] encryptData = encryptEcbPadding(keyData, contentData);
            encryptContent = new String(Base64Util.encode(encryptData), StandardCharsets.UTF_8);
        } catch (Exception e) {
            logger.error("SM4Utils.encrypt(String hexKey, String content, String charset) error {}", e.getMessage(), e);
        }
        return encryptContent;
    }

    public static byte[] encryptEcbPadding(byte[] key, byte[] data) throws Exception {
        Cipher cipher = generateEcbCipher(Cipher.ENCRYPT_MODE, key);
        return cipher.doFinal(data);
    }

    /**
     * 解密
     */
    public static String decrypt(String hexKey, String encryptContent, Charset charset) {

        String content = StringUtils.EMPTY;
        try {
            byte[] keyData = ByteUtil.hexStringToBytes(hexKey);
            byte[] encryptData = Base64.decode(encryptContent);
            byte[] contentData = decryptEcbPadding(keyData, encryptData);
            content = new String(contentData, charset);
        } catch (Exception e) {
            logger.error("SM4Utils.encrypt(String hexKey, String content, String charset) error {}", e.getMessage(), e);
        }
        return content;
    }

    private static byte[] decryptEcbPadding(byte[] key, byte[] cipherText) throws Exception {
        Cipher cipher = generateEcbCipher(Cipher.DECRYPT_MODE, key);
        return cipher.doFinal(cipherText);
    }

    private static Cipher generateEcbCipher(int mode, byte[] key) throws Exception {
        Cipher cipher = Cipher.getInstance(ALGORITHM_NAME_ECB_PADDING, BouncyCastleProvider.PROVIDER_NAME);
        Key sm4Key = new SecretKeySpec(key, ALGORITHM_NAME);
        cipher.init(mode, sm4Key);
        return cipher;
    }


    public static void main(String[] args) throws Exception {
        String hexCipher = "1231254545458545";
        String encrypt = "111222";
        System.out.println(encrypt(hexCipher, encrypt));
        System.out.println(encrypt);
        System.out.println(decrypt(hexCipher, encrypt, StandardCharsets.UTF_8));
    }

}
