package com.franklin.ideaplugin.easytesting.spring.invoke;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.franklin.ideaplugin.easytesting.core.invoke.IMethodInvoker;
import com.franklin.ideaplugin.easytesting.core.invoke.InstanceMethodInvoker;
import com.franklin.ideaplugin.easytesting.core.invoke.StaticMethodInvoker;
import com.franklin.ideaplugin.easytesting.core.log.ILogger;
import com.franklin.ideaplugin.easytesting.core.log.LoggerFactory;
import com.franklin.ideaplugin.easytesting.core.rpc.NettyServer;
import com.franklin.ideaplugin.easytesting.core.rpc.entity.ETRsp;
import com.franklin.ideaplugin.easytesting.core.rpc.entity.MethodInvokeData;
import com.franklin.ideaplugin.easytesting.core.thread.EasyTestingThreadPool;
import com.franklin.ideaplugin.easytesting.core.utils.JsonUtils;
import com.franklin.ideaplugin.easytesting.core.utils.MethodUtils;
import com.franklin.ideaplugin.easytesting.spring.constants.StrPool;
import com.franklin.ideaplugin.easytesting.spring.utils.SpringUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.LinkedHashMap;
import java.util.Objects;

/**
 * @author Ye Junhui
 * @since 2023/5/15
 */
@RequiredArgsConstructor
public class SpringMethodInvoker implements IMethodInvoker {

    private static final ILogger log = LoggerFactory.getLogger(NettyServer.class);

    private static final MediaType APPLICATION_JSON = MediaType.parse("application/json; charset=utf-8");

    private final OkHttpClient okHttpClient;

    @Value("${server.servlet.path:null}")
    private String servletPath;

    @Value("${server.servlet.context-path:null}")
    private String servletContextPath;

    @Value("${server.port:8080}")
    private String port;

    private final IMethodInvoker staticMethodInvoker;
    private final IMethodInvoker instanceMethodInvoker;

    public SpringMethodInvoker() {
        this.okHttpClient = new OkHttpClient.Builder().build();
        this.staticMethodInvoker = new StaticMethodInvoker();
        this.instanceMethodInvoker = new InstanceMethodInvoker();
    }

    @Override
    public Object invoke(MethodInvokeData methodInvokeData, Class<?> targetClass, Method targetMethod, Object[] params) throws Exception {
        if (CollectionUtil.isNotEmpty(methodInvokeData.getHeaderMap())) {
            //有请求头，发送请求
            return invokeControllerMethod(methodInvokeData);
        }
        if (MethodUtils.isStaticMethod(targetMethod)){
            return staticMethodInvoker.invoke(methodInvokeData, targetClass, targetMethod, params);
        } else {
            //无请求头直接执行
            Object bean = null;
            try {
                bean = SpringUtil.getBean(targetClass);
                return targetMethod.invoke(bean,params);
            } catch (BeansException e) {
                //没有找到bean
                //尝试实例方法
                log.info("Easy-Testing -> spring bean {} not found,try instance method ",targetClass.getName());
                return this.instanceMethodInvoker.invoke(methodInvokeData, targetClass, targetMethod, params);
            } catch (Exception e){
                log.error("Easy-Testing -> method execute fail " ,e);
                return null;
            }
        }
    }

    private Object invokeControllerMethod(MethodInvokeData methodInvokeData) {
        Request.Builder builder = new Request.Builder();
        String url = "http://localhost:" + this.port + this.getPrefix() + StrPool.INNER_REQUEST_PATH;
        builder.url(url)
                .post(RequestBody.create(APPLICATION_JSON, JsonUtils.toJSONString(methodInvokeData)));
        LinkedHashMap<String, String> headerMap = methodInvokeData.getHeaderMap();
        if (CollectionUtil.isNotEmpty(headerMap)){
            headerMap.forEach(builder::addHeader);
        }
        Request request = builder.build();
        EasyTestingThreadPool.getBizThreadPool().execute(() -> {
            Response response = null;
            try {
                response = okHttpClient.newCall(request).execute();
            } catch (IOException e) { }
            finally {
                if (Objects.nonNull(response)){
                    response.close();
                }
            }
        });

        return null;
    }

    /**
     * 前缀
     * @return
     */
    private String getPrefix(){
        if (StrUtil.isNotBlank(this.servletPath)){
            return this.servletPath;
        }
        if (StrUtil.isNotBlank(this.servletContextPath)){
            return this.servletContextPath;
        }
        return "";
    }
}
