package ru.casperix.light_ui.component.text.editor

import ru.casperix.app.surface.SurfaceProvider
import ru.casperix.app.surface.component.VirtualKeyboardApi
import ru.casperix.input.*
import ru.casperix.light_ui.component.text.TextEditorSkin
import ru.casperix.light_ui.component.text.TextHelper
import ru.casperix.light_ui.node.DrawContext
import ru.casperix.light_ui.node.InputContext
import ru.casperix.light_ui.skin.SkinProvider
import ru.casperix.light_ui.util.betweenInclusive
import ru.casperix.math.axis_aligned.float32.Box2f
import ru.casperix.math.color.Colors
import ru.casperix.math.vector.float32.Vector2f
import ru.casperix.misc.clamp
import ru.casperix.misc.sliceSafe
import ru.casperix.renderer.Renderer2D
import ru.casperix.renderer.misc.AlignMode
import ru.casperix.signals.concrete.Signal
import ru.casperix.light_ui.element.*
import ru.casperix.multiplatform.ClipBoard
import ru.casperix.multiplatform.text.TextLayout
import ru.casperix.multiplatform.text.textRenderer
import kotlin.time.Duration


class TextEditor(var text: String = "", val alignMode: AlignMode = AlignMode.CENTER_CENTER) : AbstractElement(SizeMode.fixed(Vector2f(100f))),
    ElementDrawer, ElementInput, ElementWithLayout {
    val onEnter = Signal<String>()

    var allowedSymbols: String? = null
    var prohibitedSymbols: String? = null
    var skin: TextEditorSkin = SkinProvider.skin.textEditor
    var maxLength: Int? = null

    private var isCtrl = false
    private var isShift = false
    private var isAcitve = false

    //    private var node: Node? = null
    private var cursorIndex: Int? = null
    private var selectionPhase = false
    private var selectionIndices: Pair<Int, Int>? = null
    private var textLayout = TextLayout.EMPTY

    private val cursorAnimation = TextCursorAnimation()

    override val drawer = this
    override var input: ElementInput = this

    override fun invalidateLayout() {
        textLayout = TextHelper.calculateLayout(text, placement.viewportSize, skin.label.font, alignMode)
        placement.contentArea = textLayout.getArea()
    }

    override fun draw(renderer: Renderer2D, context: DrawContext, tick: Duration): Unit = context.run {
        drawSelection(renderer, context)

        textRenderer.drawTextLayout(renderer, worldMatrix, skin.label.color, textLayout)

        if (isAcitve) {
            getCursorArea()?.let {
                if (cursorAnimation.animate(tick)) {
                    renderer.drawBox(Colors.WHITE, it, worldMatrix)
                }
            }
        }
    }

    private fun drawSelection(renderer: Renderer2D, context: DrawContext) = context.run {
        selectionIndices?.let {
            val range = IntRange.betweenInclusive(it.first, it.second)
            textLayout.symbols.sliceSafe(range).forEach {
                renderer.drawBox(Colors.BLACK, it.area, worldMatrix)
            }
        }
    }

    private fun getCursorArea(): Box2f? {
        val index = cursorIndex ?: return null
        val font = textRenderer.getPixelFont(skin.label.font)
        return TextEditorHelper.getCursorArea(index, textLayout, font.metrics.lineHeight, placement.viewportSize)
    }

    override fun input(event: InputEvent, context: InputContext) = context.run {
        if (event is KeyEvent) {
            if (event is KeyDown) {
                if (specialKeyDown(event.button)) {
                    event.captured = true
                }
            } else if (event is KeyUp) {
                specialKeyUp(event.button)
                event.captured = true
            } else if (event is KeyTyped) {
                if (!TextEditorHelper.SPECIAL_SYMBOLS.contains(event.char)) {
                    inputSymbols(event.char.toString())
                    event.captured = true
                }
            }
        } else if (event is PointerEvent) {
            if (event is PointerDown || event is PointerMove || event is PointerUp) {
                val localPosition = worldMatrix.inverse().transform(event.position)
                inputTouch(localPosition, event)
            }
        }
    }

    private fun inputTouch(localPosition: Vector2f, event: PointerEvent) {
        val isInside = localPosition.greaterOrEq(Vector2f.ZERO) && localPosition.lessOrEq(placement.viewportSize)
        if (event is PointerDown) {
            setActive(isInside)
        }

        if (isAcitve && selectionPhase) {
            val nearIndex = TextEditorHelper.getNearIndex(textLayout, localPosition)

            setupSelection(nearIndex, event)
            setCursorPosition(nearIndex ?: 0)
        }

        if (event is PointerUp) {
            selectionPhase = false
        }
    }

    fun setActive(value: Boolean) {
        val isChanged = isAcitve != value
        isAcitve = value
        if (isAcitve) {
            selectionPhase = true
            cursorIndex = null
            selectionIndices = null
        }

        if (isChanged) {
            (SurfaceProvider.surface as? VirtualKeyboardApi)?.let {
                if (isAcitve) {
                    it.showVirtualKeyboard()
                } else {
                    it.hideVirtualKeyboard()
                }
            }
        }
    }

    private fun setupSelection(nextIndex: Int?, event: PointerEvent) {
        val lastIndex = cursorIndex
        val startIndex = selectionIndices?.first ?: lastIndex
        val finishIndex = nextIndex

        selectionIndices = if (startIndex != finishIndex && startIndex != null && finishIndex != null) {
            Pair(startIndex, finishIndex)
        } else {
            null
        }
    }

    private fun specialKeyUp(button: KeyButton) {
        when (button) {
            KeyButton.CONTROL_LEFT, KeyButton.CONTROL_RIGHT -> {
                isCtrl = false
            }

            KeyButton.SHIFT_LEFT, KeyButton.SHIFT_RIGHT -> {
                isShift = false
            }

            else -> {

            }
        }
    }

    private fun specialKeyDown(button: KeyButton): Boolean {

        return when (button) {
            KeyButton.CONTROL_LEFT, KeyButton.CONTROL_RIGHT -> {
                isCtrl = true
                true
            }

            KeyButton.SHIFT_LEFT, KeyButton.SHIFT_RIGHT -> {
                isShift = true
                true
            }

            KeyButton.C -> {
                if (isCtrl) {
                    ClipBoard.copyText(text)
                }
                true
            }

            KeyButton.INSERT -> {
                if (isCtrl) {
                    ClipBoard.copyText(text)
                } else if (isShift) {
                    ClipBoard.pasteText()?.let {
                        inputSymbols(it)
                    }
                }
                true
            }

            KeyButton.V -> {
                if (isCtrl) {
                    ClipBoard.pasteText()?.let {
                        inputSymbols(it)
                    }
                }
                true
            }

            KeyButton.ARROW_LEFT -> {
                val cursorIndex = cursorIndex ?: return false
                setCursorPosition(cursorIndex - 1)
                true
            }

            KeyButton.ARROW_RIGHT -> {
                val cursorIndex = cursorIndex ?: return false
                setCursorPosition(cursorIndex + 1)
                true
            }

            KeyButton.ESCAPE -> {
                isAcitve = false
                true
            }

            KeyButton.ENTER, KeyButton.NUMPAD_ENTER -> {
                onEnter.set(text)
                true
            }

            KeyButton.BACKSPACE -> {
                removeBackward()
            }

            KeyButton.DELETE -> {
                removeForward()
            }

            else -> {
                false
            }
        }
    }

    private fun removeBackward(): Boolean {
        if (removeSelectionIfExist()) return true

        val cursorIndex = cursorIndex ?: return false
        val lastText = text
        if (cursorIndex <= 0) return true
        setupText(lastText.sliceSafe(0 until cursorIndex - 1) + lastText.sliceSafe(cursorIndex..lastText.length))
        setCursorPosition(cursorIndex - 1)
        return true
    }

    private fun removeForward(): Boolean {
        if (removeSelectionIfExist()) return true

        val cursorIndex = cursorIndex ?: return false

        val lastText = text
        if (cursorIndex >= lastText.length) return true
        setupText(lastText.sliceSafe(0 until cursorIndex) + lastText.sliceSafe(cursorIndex + 1..lastText.length))
        return true
    }

    private fun removeSelectionIfExist(): Boolean {
        val selection = selectionIndices ?: return false

        val range = IntRange.betweenInclusive(selection.first, selection.second)
        setupText(text.sliceSafe(0 until range.start) + text.sliceSafe((range.endInclusive + 1) until text.length))
        selectionIndices = null
        setCursorPosition(range.start)
        return true
    }

    private fun inputSymbols(inputSymbols: String) {
        val symbols = filterText(inputSymbols)
        if (symbols.isEmpty()) return

        removeSelectionIfExist()

        val lastCursorIndex = cursorIndex ?: 0
        val lastText = text
        val nextText =
            lastText.sliceSafe(0 until lastCursorIndex) + symbols + lastText.sliceSafe(lastCursorIndex..lastText.length)


        setupText(nextText)
        setCursorPosition((cursorIndex ?: 0) + symbols.length)
    }

    fun setCursorPosition(index: Int?) {
        cursorAnimation.reset()
        cursorIndex = index?.clamp(0, text.length + 1)
    }

    private fun setupText(value: String) {
        var next = value
        maxLength?.let {
            next = next.take(it)
        }
        text = next
    }


    private fun filterText(inputSymbols: String): String {
        val allowedSymbols = allowedSymbols
        val prohibitedSymbols = prohibitedSymbols

        return if (allowedSymbols == null && prohibitedSymbols == null) {
            inputSymbols
        } else {
            inputSymbols.filter {
                (allowedSymbols != null && allowedSymbols.contains(it)) || (prohibitedSymbols != null && !prohibitedSymbols.contains(
                    it
                ))
            }
        }
    }

}


