package ru.casperix.multiplatform.loader

import ru.casperix.renderer.pixel_map.PixelMap
import ru.casperix.signals.concrete.EitherFuture
import ru.casperix.signals.concrete.EitherSignal
import ru.casperix.multiplatform.loader.JvmImageConverter.createPixelMap
import java.io.File
import javax.imageio.ImageIO

actual val resourceLoader: AbstractResourceLoader = JvmLoaders()

class JvmLoaders : AbstractResourceLoader {

    override fun saveImage(path: String, pixelMap: PixelMap): ResourceSaveError? {
        val image = JvmImageConverter.createBufferedImage(pixelMap)
        val file = File(path)
        ImageIO.write(image, "png", file)
        return null
    }

    override fun loadBytes(path: String): EitherFuture<ByteArray, ResourceLoadError> {
        val loader = EitherSignal<ByteArray, ResourceLoadError>()
        val classloader = Thread.currentThread().getContextClassLoader()
        val stream = classloader.getResourceAsStream(path)

        loader.apply {
            if (stream == null) {
                reject(ResourceNotFoundLoadError(path))
            } else {
                accept(stream.readBytes())
            }
        }
        return loader
    }

    override fun loadImage(path: String): EitherFuture<PixelMap, ResourceLoadError> {
        val loader = EitherSignal<PixelMap, ResourceLoadError>()
        try {
            val classloader = Thread.currentThread().getContextClassLoader()
            val url = classloader.getResource(path)
            if (url === null) {
                loader.reject(ResourceNotFoundLoadError(path))
                return loader
            }
            val image = ImageIO.read(url)
            val pixelMap = createPixelMap(image, path)

            loader.accept(pixelMap)
        } catch (e: Throwable) {
            println("Can't loadPNG: ${e.message}")
            loader.reject(ResourceCustomLoadError(e.message ?: ""))
        }
        return loader
    }
}

