package casperix.font.pixel

import casperix.font.FontReference
import casperix.graphic.pixel_map.PixelMapRegion
import casperix.loader.ResourceLoader
import casperix.math.axis_aligned.int32.Box2i
import casperix.math.color.Colors
import casperix.math.vector.int32.Vector2i
import casperix.math.rectangle_packer.RectanglePacker
import casperix.math.rectangle_packer.RectangleSource
import casperix.renderer.pixel_map.PixelMap
import casperix.renderer.text.TextRenderConfig.debugDrawFontAtlas
import casperix.renderer.text.TextRenderConfig.debugSaveFontAtlas
import casperix.util.PixelMapDrawer

@ExperimentalUnsignedTypes
object PixelFontPacker {
    fun packFont(font: PixelFont): PixelFont {
        val symbols = font.symbols
        val withGraphic = symbols.filter { it.graphic != null }
        val withoutGraphic = symbols.filter { it.graphic == null }

        val packedSymbols = packGraphicSymbols(font.reference, withGraphic)
        return PixelFont(font.reference, font.metrics, packedSymbols + withoutGraphic)
    }

    private fun packGraphicSymbols(reference: FontReference, symbols: Collection<PixelFontSymbol>): Collection<PixelFontSymbol> {
        val containers = symbols.map {
            RectangleSource(it, it.size)
        }

        val packed = RectanglePacker.pack(containers)
            ?: throw Exception("Cant pack items")

        val atlas = PixelMap.RGBA(packed.dimension)

        packed.containers.forEach { container ->
            val pixelRegion = container.source.graphic
                ?: throw Exception("Only graphic symbol must packed to atlas")

            if (debugDrawFontAtlas) {
                PixelMapDrawer.fillColor(
                    atlas,
                    container.position,
                    Colors.RED.toColor4b(),
                    pixelRegion.dimension,
                    false
                )
                PixelMapDrawer.fillColor(
                    atlas,
                    container.position + Vector2i.ONE,
                    Colors.BLUE.toColor4b(),
                    pixelRegion.dimension - Vector2i(2),
                    false
                )
            }
            PixelMapDrawer.drawImage(
                atlas,
                container.position,
                pixelRegion.atlas,
                Vector2i.ZERO,
                pixelRegion.region.dimension,
                debugDrawFontAtlas
            )
        }

        val packedSymbols = packed.containers.map { container ->
            val original = container.source
            val graphic = PixelMapRegion(atlas, Box2i.byDimension(container.position, original.size))
            original.copy(graphic = graphic)
        }
        if (debugSaveFontAtlas) {
            val postfix = "_" + reference.name + "_" + reference.size
            ResourceLoader.saveImage("font_atlas$postfix.png", atlas)
        }
        return packedSymbols
    }
}