package casperix.renderer.text.impl

import casperix.font.FontReference
import casperix.font.localize.CharacterSets
import casperix.font.pixel.PixelFont
import casperix.font.pixel.PixelFontPacker
import casperix.math.color.Color
import casperix.math.quad_matrix.float32.Matrix3f
import casperix.math.vector.float32.Vector2f
import casperix.renderer.Renderer2D
import casperix.renderer.text.TextLayout
import casperix.renderer.text.TextRenderConfig
import casperix.renderer.text.TextRendererApi

@ExperimentalUnsignedTypes
abstract class AbstractTextRenderer : TextRendererApi {
    abstract fun buildFont(reference: FontReference, chars: Set<Char>): PixelFont

    private val fontCache = mutableMapOf<FontReference, PixelFont>()
    private var cacheConfig = TextRenderConfig.actualLanguages


    override fun getPixelFont(reference: FontReference): PixelFont {
        if (cacheConfig != TextRenderConfig.actualLanguages) {
            cacheConfig = TextRenderConfig.actualLanguages
            fontCache.clear()
        }
        return fontCache.getOrPut(reference) {
            val alphabetChars = CharacterSets.all
                .filter { cacheConfig.contains(it.language) }
                .flatMap { it.charSet }
                .toSet()
            val font = buildFont(reference, CharacterSets.COMMON_CHARACTER_SET + alphabetChars)
            PixelFontPacker.packFont(font)
        }
    }


    override fun createTextLayout(text: String, areaSize: Vector2f, font: PixelFont): TextLayout {
        return TextLayoutBuilder.layout(text, areaSize, font)
    }


    override fun Renderer2D.drawText(
        text: String,
        reference: FontReference,
        matrix: Matrix3f,
        availableArea: Vector2f,
        color: Color,
        backgroundColor: Color?
    ): TextLayout {
        val pixelFont = getPixelFont(reference)
        val layout = createTextLayout(text, availableArea, pixelFont)
        TextLayoutRender.drawTextLayout(this, matrix, color, layout)
        return layout
    }

    override fun Renderer2D.drawTextLayout(matrix: Matrix3f, color: Color, layout: TextLayout) {
        TextLayoutRender.drawTextLayout(this, matrix, color, layout)
    }

    override fun Renderer2D.drawTextLayoutMetrics(matrix: Matrix3f, font: PixelFont, layout: TextLayout) {
        TextLayoutRender.drawTextLayoutMetrics(this, matrix, font, layout)
    }

}