package casperix.renderer.text.impl

import casperix.dimension
import casperix.font.pixel.PixelFont
import casperix.math.color.Color
import casperix.math.color.Colors
import casperix.math.quad_matrix.float32.Matrix3f
import casperix.math.straight_line.float32.LineSegment2f
import casperix.math.vector.float32.Vector2f
import casperix.math.vector.int32.Vector2i
import casperix.renderer.Renderer2D
import casperix.renderer.material.Material
import casperix.renderer.pixel_map.PixelMap
import casperix.renderer.text.TextLayout
import casperix.renderer.text.TextRenderConfig
import casperix.renderer.vector.builder.VectorBuilder
import kotlin.math.roundToInt

@ExperimentalUnsignedTypes
object TextLayoutRender {

    fun drawTextLayoutMetrics(
        renderer: Renderer2D,
        matrix: Matrix3f,
        font: PixelFont,
        layout: TextLayout
    ) = renderer.run {
        val area = layout.getArea()
        drawBox(Colors.BLACK.setAlpha(0.2f), area, matrix)

        val lineThick = 1f
        val alpha = 0.6f

        layout.symbols.forEach { symbolLayout ->
            val xAxis = symbolLayout.symbol.size.xAxis.toVector2f()
            val baseLinePivot = symbolLayout.pivot

            val ascentLine = LineSegment2f.byDelta(baseLinePivot - Vector2f(0f, font.metrics.ascent), xAxis)
            val baseLine = LineSegment2f.byDelta(baseLinePivot, xAxis)
            val descentLine = LineSegment2f.byDelta(baseLinePivot + Vector2f(0f, font.metrics.descent), xAxis)
            val bottomLine = LineSegment2f.byDelta(baseLinePivot + Vector2f(0f, font.metrics.descent + font.metrics.leading), xAxis)

            drawSegment(Colors.GREEN.setAlpha(alpha), matrix.transform(ascentLine), lineThick)
            drawSegment(Colors.RED.setAlpha(alpha), matrix.transform(baseLine), lineThick)
            drawSegment(Colors.BLUE.setAlpha(alpha), matrix.transform(descentLine), lineThick)
            drawSegment(Colors.YELLOW.setAlpha(alpha), matrix.transform(bottomLine), lineThick)
        }

    }

    fun drawTextLayout(
        renderer: Renderer2D,
        matrix: Matrix3f,
        color: Color,
        layout: TextLayout
    ) {
        var lastAtlas: PixelMap? = null
        var lastMaterial = Material()

        val graphicList = layout.symbols.mapNotNull { symbolLayout ->
            val graphic = symbolLayout.symbol.graphic
            if (graphic != null) {
                Pair(symbolLayout.pivot + symbolLayout.symbol.baselineOffset.toVector2f(), graphic)
            } else null
        }

        val builder = VectorBuilder(hasPosition2 = true, hasTextureCoord = true)

        graphicList.forEachIndexed { index, (position, pixelRegion) ->
            val atlas = pixelRegion.atlas
            val atlasSize = atlas.dimension().toVector2f()
            val sizeFloat = pixelRegion.region.dimension.toVector2f()
            val childMatrix = Matrix3f.translate(position)
            val summaryMatrix = childMatrix * matrix

            if (TextRenderConfig.textRoundToPixel) {
                summaryMatrix.roundTranslate()
            }

            if (lastAtlas == null) {
                lastMaterial = Material(color, albedoMap = atlas)
                lastAtlas = atlas
            } else if (lastAtlas != atlas) {
                throw Exception("Please pack font with PixelFontPacker")
            }

            val tex00 = pixelRegion.region.min.toVector2f() / atlasSize
            val tex11 = (pixelRegion.region.max + Vector2i.ONE).toVector2f() / atlasSize
            val tex10 = Vector2f(tex11.x, tex00.y)
            val tex01 = Vector2f(tex00.x, tex11.y)

            val pos00 = summaryMatrix.transform(Vector2f.ZERO)
            val pos10 = summaryMatrix.transform(sizeFloat.xAxis)
            val pos11 = summaryMatrix.transform(sizeFloat)
            val pos01 = summaryMatrix.transform(sizeFloat.yAxis)

            builder.addQuad(pos00, pos10, pos11, pos01, tex00, tex10, tex11, tex01)
        }
        val material = lastMaterial

        renderer.drawGraphic(builder.buildGraphic(material))
    }

    private fun Matrix3f.roundTranslate() {
        data[6] = data[6].roundToInt().toFloat()
        data[7] = data[7].roundToInt().toFloat()
    }
}