package casperix.loader

import casperix.renderer.pixel_map.PixelMap
import casperix.signals.concrete.EitherFuture
import casperix.signals.concrete.EitherSignal
import casperix.loader.JvmImageConverter.createPixelMap
import java.io.File
import java.io.IOException
import javax.imageio.ImageIO


@OptIn(ExperimentalUnsignedTypes::class)
actual object ResourceLoader : AbstractResourceLoader {

    override fun saveImage(path: String, pixelMap: PixelMap): ResourceSaveError? {
        val image = JvmImageConverter.createBufferedImage(pixelMap)
        val file = File(path)
        ImageIO.write(image, "png", file)
        return null
    }

    override fun load(path: String): EitherFuture<ByteArray, ResourceLoadError> {
        val loader = EitherSignal<ByteArray, ResourceLoadError>()
        val classloader = Thread.currentThread().getContextClassLoader()
        val stream = classloader.getResourceAsStream(path)

        loader.apply {
            if (stream == null) {
                reject(ResourceNotFoundLoadError)
            } else {
                accept(stream.readBytes())
            }
        }
        return loader
    }

    override fun loadImage(path: String): EitherFuture<PixelMap, ResourceLoadError> {
        val loader = EitherSignal<PixelMap, ResourceLoadError>()
        try {
            val classloader = Thread.currentThread().getContextClassLoader()
            val url = classloader.getResource(path)
            val image = ImageIO.read(url)
            val pixelMap = createPixelMap(image)

            loader.accept(pixelMap)
        } catch (e: IOException) {
            println("Can't loadPNG: ${e.message}")
            loader.reject(ResourceCustomLoadError(e.message ?: ""))
        }
        return loader
    }
}

