package ru.casperix.simple_console.ui

import ru.casperix.input.InputEvent
import ru.casperix.input.KeyButton
import ru.casperix.input.KeyDown
import ru.casperix.input.KeyUp
import ru.casperix.light_ui.element.ElementInput
import ru.casperix.light_ui.element.ElementUpdate
import ru.casperix.light_ui.node.InputContext
import ru.casperix.misc.sliceSafe
import ru.casperix.simple_console.article.ArticleFormat.asRaw
import ru.casperix.simple_console.article.NameValueSection
import ru.casperix.simple_console.core.ConsoleLogic

class ConsoleViewController(val logic: ConsoleLogic, val view: ConsoleView = ConsoleView()) : ElementInput {
    private var firstFocus = false
    private var historyIndex = 0

    init {
        view.apply {
            logic.outputSignal.then {
                outputText.text = it
            }
            logic.actionStarted.then {
                commandInfo.text = listOf(NameValueSection("last input", it.input.asRaw())).asRaw()
            }

            attachInput()

            ElementUpdate {
                update()
            }
        }
        autocompleteHelper()
    }

    private fun attachInput() {
        val self = view.inputField.input
        view.inputField.input = object : ElementInput {
            override fun input(event: InputEvent, context: InputContext) {
                self.input(event, context)
                this@ConsoleViewController.input(event, context)
            }
        }
    }


    override fun input(event: InputEvent, context: InputContext) {
        event.captured = if (event is KeyDown) {
            keyDownHandler(event.button)
        } else if (event is KeyUp) {
            keyUpHandler(event.button)
        } else {
            false
        } || event.captured
    }

    private fun keyDownHandler(button: KeyButton): Boolean {
        return when (button) {
            KeyButton.ENTER, KeyButton.NUMPAD_ENTER -> {
                autocompleteInput()
                enter()
                true
            }

            KeyButton.TAB -> {
                autocompleteInput()
                true
            }

            KeyButton.ARROW_UP -> {
                moveByHistory(true)
                true
            }

            KeyButton.ARROW_DOWN -> {
                moveByHistory(false)
                true
            }

            else -> {
                false
            }
        }
    }

    private fun moveByHistory(up: Boolean) {
        val history = logic.getHistory()
        if (history.isEmpty()) return

        if (up) historyIndex--
        else historyIndex++

        historyIndex = historyIndex.coerceIn(history.indices)

        view.inputField.text = history.get(historyIndex).input.asRaw()
    }

    private fun keyUpHandler(button: KeyButton): Boolean {
        autocompleteHelper()
        return false
    }

    private fun autocompleteHelper() = view.apply {
        val variants = logic.autocompleteCommand(inputField.text)

        val text = if (variants.isEmpty()) {
            //TODO
//            val style  = ConfigProvider.config.ui.style
//            colorizeText("waiting input...", style.nameColor)
            "waiting input..."
        } else {
            variants.sliceSafe(0..2).mapIndexed { index, value ->
                val prefix = if (index == 0) "> " else "  "
                prefix + value.original
            }.joinToString("\n")
        }
        helperArea.text = text
    }

    private fun autocompleteInput() = view.run {
        val variants = logic.autocompleteCommand(inputField.text)
        if (variants.isEmpty()) return

        inputField.text = variants.first().candidate.asRaw()
        inputField.setCursorPosition(inputField.text.length)
    }

    private fun enter() = view.apply {
        historyIndex = Int.MAX_VALUE
        logic.launchAction(inputField.text)
        inputField.text = ""
    }

    private fun updateFocus() {
        if (firstFocus) return
        firstFocus = true
        view.inputField.setActive(true)
    }

    fun update() {
        updateFocus()
        logic.update()
    }

}