package ru.casperix.simple_console.core

import ru.casperix.misc.format.StringUtil.camelCaseSplit
import ru.casperix.misc.sliceSafe
import kotlin.math.max
import kotlin.math.min


object CommandParser {

    val INPUT_PARSER_REG = Regex("\\S+")

    class CommandScore(val original: String, val candidate: CommandInput, val score: Int)

    fun autocompleteCommand(commandNames: Collection<String>, raw: String): List<CommandScore> {
        val input = parseInput(raw) ?: return emptyList()

        return commandNames.map { name ->
            CommandScore(name, CommandInput(name, input.arguments), getLikeScore(name, input.name))
        }.filter { it.score > 0 }.sortedByDescending { it.score }

    }

    fun parseInput(raw: String): CommandInput? {
        val words = INPUT_PARSER_REG.findAll(raw).toList().map { it.value }
        if (words.isEmpty()) return null
        return CommandInput(words.first(), words.sliceSafe(1 until words.size))
    }

    /**
     * 100 -- equal
     * (0-100) -- like
     * 0 -- not equal
     */
    fun getLikeScore(name: String, input: String): Int {
        if (name == input) return 100
        if (name.lowercase() == input.lowercase()) return 90

        getCamelCaseScore(name, input)?.let {
            return it
        }
        getCamelCaseScore(name, input.uppercase())?.let {
            return it / 2
        }

        if (input.isNotEmpty() && name.startsWith(input, true)) return 1
        return 0
    }

    private fun getCamelCaseScore(name: String, input: String): Int? {
        val nameItems = camelCaseSplit(name)
        val inputItems = camelCaseSplit(input)
        val partAmount = min(nameItems.size, inputItems.size)
        val maxPartAmount = max(nameItems.size, inputItems.size)
        if (partAmount > 0) {
            var score = 0
            repeat(partAmount) {
                val namePart = nameItems.getOrNull(it)
                val inputPart = inputItems.getOrNull(it)
                if (namePart != null && inputPart != null) {
                    if (namePart.startsWith(inputPart)) {
                        score += 10
                    } else {
                        score -= 10
                    }
                }
            }
            if (score > 0) {
                return (score / maxPartAmount).coerceIn(0, 80)
            }
        }
        return null
    }


}

