/*
 * Copyright (C) 2024 Berner Fachhochschule https://e-voting.bfh.ch
 *
 *  - This program is free software: you can redistribute it and/or modify                           -
 *  - it under the terms of the GNU Affero General Public License as published by                    -
 *  - the Free Software Foundation, either version 3 of the License, or                              -
 *  - (at your option) any later version.                                                            -
 *  -                                                                                                -
 *  - This program is distributed in the hope that it will be useful,                                -
 *  - but WITHOUT ANY WARRANTY; without even the implied warranty of                                 -
 *  - MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                                   -
 *  - GNU General Public License for more details.                                                   -
 *  -                                                                                                -
 *  - You should have received a copy of the GNU Affero General Public License                       -
 *  - along with this program. If not, see <http://www.gnu.org/licenses/>.                           -
 */
package ch.openchvote.algorithms.general.algorithms;

import ch.openchvote.algorithms.Algorithm;
import ch.openchvote.algorithms.parameters.security.NIZKPParameters;
import ch.openchvote.utilities.sequence.Vector;
import ch.openchvote.utilities.set.IntSet;

import java.math.BigInteger;

/**
 * Implementation of Algorithm 8.5 from CHVote Protocol Specification
 */
public final class GetChallenges extends Algorithm<Vector<BigInteger>> {

    @SuppressWarnings({"MissingJavadoc", "UnnecessaryLocalVariable"})
    static public //
    <SP extends NIZKPParameters>
    Vector<BigInteger>
    run(int n, Object y, SP securityParameters) {

        // SECURITY PARAMETERS
        Precondition.checkNotNull(securityParameters);
        var twoToTheTau = securityParameters.get_twoToTheTau();

        // PRECONDITIONS
        Precondition.checkNotNull(y);
        Precondition.check(IntSet.NN.contains(n));

        // PREPARATION
        var builder_bold_c = new Vector.Builder<BigInteger>(n);

        // ALGORITHM
        var H = RecHash.run(y, securityParameters);
        for (int i : IntSet.range(1, n)) {
            var H_i = RecHash.run(H, i, securityParameters);
            var c_i = ByteArrayToInteger.run(H_i).mod(twoToTheTau);
            builder_bold_c.set(i, c_i);
        }
        var bold_c = builder_bold_c.build();
        return bold_c;
    }

}
