/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.configure;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.annotation.EnableGuice;
import org.cattleframework.aop.annotation.GuiceBindBean;
import org.cattleframework.aop.annotation.GuiceModule;
import org.cattleframework.aop.annotation.SpringBindBean;
import org.cattleframework.aop.context.SpringContext;
import org.cattleframework.aop.guice.GuiceBeanFactory;
import org.cattleframework.aop.guice.SpringModule;
import org.cattleframework.aop.reflections.ReflectionsFactory;
import org.cattleframework.aop.utils.SimpleReflectUtils;
import org.cattleframework.exception.CommonException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.AutowireCandidateQualifier;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.boot.autoconfigure.AutoConfigurationPackages;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.ResolvableType;
import org.springframework.core.annotation.AnnotationAttributes;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;

import com.google.inject.Binding;
import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Key;
import com.google.inject.Module;
import com.google.inject.Scopes;
import com.google.inject.Stage;
import com.google.inject.name.Named;
import com.google.inject.spi.ElementSource;

class AopBeanRegistrar implements ImportBeanDefinitionRegistrar, BeanFactoryAware, EnvironmentAware {

    private static final Logger logger = LoggerFactory.getLogger(AopBeanRegistrar.class);

    private static final String DEFAULT_PACKAGE = "org.cattleframework";

    private static final String COMPONENT_SCAN = "cattle.component-scan";

    private ConfigurableListableBeanFactory beanFactory;

    private Environment environment;

    @Override
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
	if (beanFactory instanceof ConfigurableListableBeanFactory listableBeanFactory) {
	    this.beanFactory = listableBeanFactory;
	}
    }

    @Override
    public void setEnvironment(Environment environment) {
	this.environment = environment;
    }

    @Override
    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
	if (this.beanFactory == null) {
	    return;
	}
	String[] scanPackages = processComponentScan(registry);
	ReflectionsFactory.initialize(scanPackages);
	boolean enableGuice = searchEnableGuice(registry);
	if (enableGuice) {
	    String guiceBeanFactoryClassName = GuiceBeanFactory.class.getName();
	    if (!registry.containsBeanDefinition(guiceBeanFactoryClassName)) {
		Injector injector = createInjector();
		BeanDefinition guiceBeanFactoryDefinition = BeanDefinitionBuilder
			.rootBeanDefinition(GuiceBeanFactory.class).addConstructorArgValue(injector)
			.setRole(BeanDefinition.ROLE_INFRASTRUCTURE).setScope(BeanDefinition.SCOPE_SINGLETON)
			.getBeanDefinition();
		registry.registerBeanDefinition(guiceBeanFactoryClassName, guiceBeanFactoryDefinition);
		Map<Key<?>, Binding<?>> bindings = injector.getAllBindings();
		mapBindings(injector, bindings, (BeanDefinitionRegistry) beanFactory);
	    }
	}
	String springContextClassName = SpringContext.class.getName();
	if (!registry.containsBeanDefinition(springContextClassName)) {
	    BeanDefinition springContextDefinition = BeanDefinitionBuilder.rootBeanDefinition(SpringContext.class)
		    .addConstructorArgValue(beanFactory).addConstructorArgValue(environment)
		    .setRole(BeanDefinition.ROLE_INFRASTRUCTURE).setScope(BeanDefinition.SCOPE_SINGLETON)
		    .getBeanDefinition();
	    registry.registerBeanDefinition(springContextClassName, springContextDefinition);
	}
    }

    private boolean searchEnableGuice(BeanDefinitionRegistry registry) {
	String[] beanNames = registry.getBeanDefinitionNames();
	return Arrays.stream(beanNames).anyMatch(beanName -> {
	    BeanDefinition beanDefinition = registry.getBeanDefinition(beanName);
	    if (beanDefinition instanceof AnnotatedBeanDefinition) {
		AnnotatedBeanDefinition annotatedBeanDefinition = (AnnotatedBeanDefinition) beanDefinition;
		return annotatedBeanDefinition.getMetadata().hasAnnotation(EnableGuice.class.getName());
	    }
	    return false;
	});
    }

    private String[] processComponentScan(BeanDefinitionRegistry registry) {
	List<String> basePackages = AutoConfigurationPackages.get(beanFactory);
	Set<String> springPackages = new HashSet<String>();
	String[] beanNames = registry.getBeanDefinitionNames();
	Arrays.stream(beanNames).forEach(beanName -> {
	    BeanDefinition beanDefinition = registry.getBeanDefinition(beanName);
	    if (beanDefinition instanceof AnnotatedBeanDefinition) {
		AnnotatedBeanDefinition annotatedBeanDefinition = (AnnotatedBeanDefinition) beanDefinition;
		AnnotationAttributes componentScanAttributes = AnnotationAttributes.fromMap(annotatedBeanDefinition
			.getMetadata().getAnnotationAttributes(ComponentScan.class.getName(), true));
		if (componentScanAttributes != null) {
		    String[] packages = componentScanAttributes.getStringArray("basePackages");
		    if (ArrayUtils.isNotEmpty(packages)) {
			Arrays.stream(packages).forEach(strPackage -> {
			    if (StringUtils.isNotBlank(strPackage)) {
				springPackages.add(strPackage.trim());
			    }
			});
		    }
		    String[] basePackageClasses = componentScanAttributes.getStringArray("basePackageClasses");
		    if (ArrayUtils.isNotEmpty(basePackageClasses)) {
			Arrays.stream(basePackageClasses).forEach(basePackageClass -> {
			    if (StringUtils.isNotBlank(basePackageClass)) {
				springPackages.add(
					org.apache.commons.lang3.ClassUtils.getPackageName(basePackageClass.trim()));
			    }
			});
		    }
		}
	    }
	});
	Set<String> configPackages = new HashSet<String>();
	String strConfigComponentScan = environment.getProperty(COMPONENT_SCAN);
	if (StringUtils.isNotBlank(strConfigComponentScan)) {
	    String[] configComponentScanArray = StringUtils.split(strConfigComponentScan, ";");
	    if (ArrayUtils.isNotEmpty(configComponentScanArray)) {
		Arrays.stream(configComponentScanArray).forEach(configComponentScan -> {
		    if (StringUtils.isNotBlank(configComponentScan)) {
			configPackages.add(configComponentScan.trim());
		    }
		});
	    }
	}
	logger.debug("-------------------- 自动扫描组件的信息 --------------------");
	Set<String> scanedPackages = new HashSet<String>();
	if (CollectionUtils.isNotEmpty(basePackages)) {
	    logger.debug("基础扫描的包名:{}", basePackages);
	    scanedPackages.addAll(basePackages);
	}
	if (CollectionUtils.isNotEmpty(springPackages)) {
	    logger.debug("Spring扫描的包名:{}", springPackages);
	    scanedPackages.addAll(springPackages);
	}
	if (CollectionUtils.isNotEmpty(configPackages)) {
	    logger.debug("配置定义扫描的包名:{}", configPackages);
	    scanedPackages.addAll(configPackages);
	}
	logger.debug("------------------------------------------------------------");
	scanedPackages.add(DEFAULT_PACKAGE);
	return scanedPackages.toArray(new String[0]);
    }

    private Injector createInjector() {
	Set<Module> modules = new HashSet<Module>();
	Set<Class<?>> moduleClasses = ReflectionsFactory.getTypesAnnotatedWith(GuiceModule.class);
	moduleClasses.stream().forEachOrdered(item -> {
	    if (SimpleReflectUtils.implementsInterface(item, Module.class) && SimpleReflectUtils.isClass(item)) {
		try {
		    Object obj = SimpleReflectUtils.instance(item);
		    if (obj != null) {
			logger.debug("找到Guice模块,类:{}", item.getName());
			modules.add((Module) obj);
		    }
		} catch (CommonException e) {
		}
	    }
	});
	Collection<Module> sModules = beanFactory.getBeansOfType(Module.class).values();
	sModules.stream().forEach(item -> {
	    Class<? extends Module> clazz = item.getClass();
	    if (clazz.getAnnotation(GuiceModule.class) == null) {
		logger.debug("找到Guice模块,类:{}", clazz.getName());
		modules.add(item);
	    }
	});
	modules.add(new SpringModule(beanFactory));
	return Guice.createInjector(Stage.PRODUCTION, modules);
    }

    private void mapBindings(Injector injector, Map<Key<?>, Binding<?>> bindings, BeanDefinitionRegistry registry) {
	for (Entry<Key<?>, Binding<?>> entry : bindings.entrySet()) {
	    Key<?> key = entry.getKey();
	    Class<?> beanType = key.getTypeLiteral().getRawType();
	    if (Stage.class.equals(beanType) || java.util.logging.Logger.class.equals(beanType)
		    || beanType.getAnnotation(GuiceBindBean.class) != null || java.util.Set.class.equals(beanType)
		    || java.util.Collection.class.equals(beanType) || Injector.class.equals(beanType)
		    || beanType.getAnnotation(SpringBindBean.class) == null) {
		continue;
	    }
	    if (SpringModule.CATTLE_GUICE_SOURCE
		    .equals(Optional.ofNullable(entry.getValue().getSource()).map(Object::toString).orElse(""))) {
		continue;
	    }
	    if (key.getAnnotationType() != null
		    && key.getAnnotationType().getName().equals("com.google.inject.internal.Element")) {
		continue;
	    }
	    Binding<?> binding = entry.getValue();
	    Object source = binding.getSource();
	    if (source != null && source instanceof ElementSource) {
		if (((ElementSource) source).getModuleClassNames().contains(SpringModule.class.getName())) {
		    continue;
		}
	    }
	    RootBeanDefinition bean = new RootBeanDefinition(GuiceFactoryBean.class);
	    MutablePropertyValues mutablePropertyValues = new MutablePropertyValues();
	    mutablePropertyValues.add("injector", injector);
	    mutablePropertyValues.add("key", key);
	    mutablePropertyValues.add("beanType", beanType);
	    mutablePropertyValues.add("singleton", Scopes.isSingleton(binding));
	    bean.setPropertyValues(mutablePropertyValues);
	    bean.setTargetType(ResolvableType.forType(key.getTypeLiteral().getType()));
	    if (!Scopes.isSingleton(binding)) {
		bean.setScope(ConfigurableBeanFactory.SCOPE_PROTOTYPE);
	    }
	    if (source instanceof ElementSource) {
		bean.setResourceDescription(((ElementSource) source).getDeclaringSource().toString());
	    } else {
		bean.setResourceDescription(SpringModule.CATTLE_GUICE_SOURCE);
	    }
	    bean.setAttribute(SpringModule.CATTLE_GUICE_SOURCE, true);
	    if (key.getAnnotationType() != null) {
		bean.addQualifier(new AutowireCandidateQualifier(Qualifier.class, getValueAttributeForNamed(key)));
		bean.addQualifier(
			new AutowireCandidateQualifier(key.getAnnotationType(), getValueAttributeForNamed(key)));
	    }
	    registry.registerBeanDefinition(extractName(key), bean);
	}
    }

    private String extractName(Key<?> key) {
	final String className = key.getTypeLiteral().getType().getTypeName();
	String valueAttribute = getValueAttributeForNamed(key);
	if (valueAttribute != null) {
	    return valueAttribute + "_" + className;
	} else {
	    return className;
	}
    }

    private String getValueAttributeForNamed(Key<?> key) {
	if (key.getAnnotation() instanceof Named) {
	    return ((Named) key.getAnnotation()).value();
	} else if (key.getAnnotation() instanceof jakarta.inject.Named) {
	    return ((jakarta.inject.Named) key.getAnnotation()).value();
	} else if (key.getAnnotationType() != null) {
	    return key.getAnnotationType().getName();
	} else {
	    return null;
	}
    }
}