/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.guice;

import java.util.Set;

import org.cattleframework.aop.reflections.ReflectionsFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.inject.AbstractModule;
import com.google.inject.Singleton;
import com.google.inject.binder.ScopedBindingBuilder;
import com.google.inject.multibindings.Multibinder;

/**
 * Guice模块抽象
 * 
 * @author orange
 *
 */
public abstract class AbstractGuiceModule extends AbstractModule {

    private static final Logger logger = LoggerFactory.getLogger(AbstractGuiceModule.class);

    @SuppressWarnings("rawtypes")
    protected void bindClass(Class clazz) {
	bindClass(clazz, true);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    protected void bindClass(Class clazz, boolean singleton) {
	Class bindClass = ReflectionsFactory.getSubTypesOfByLoadLevel(clazz);
	if (bindClass != null) {
	    ScopedBindingBuilder scopedBindingBuilder = bind(clazz).to(bindClass);
	    if (singleton) {
		scopedBindingBuilder.in(Singleton.class);
	    }
	    logger.debug("注入Guice,接口类:{},实现类:{},单例:{}", clazz.getName(), bindClass.getName(), singleton);
	}
    }

    @SuppressWarnings("rawtypes")
    protected void bindMultiClass(Class clazz) {
	bindMultiClass(clazz, true);
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    protected void bindMultiClass(Class clazz, boolean singleton) {
	Multibinder binder = Multibinder.newSetBinder(binder(), clazz);
	Set<Class> classes = ReflectionsFactory.getSubTypesOf(clazz);
	classes.stream().forEach(item -> {
	    ScopedBindingBuilder scopedBindingBuilder = binder.addBinding().to(item);
	    if (singleton) {
		scopedBindingBuilder.in(Singleton.class);
	    }
	    logger.debug("注入Guice,类:{},单例:{}", item.getName(), singleton);
	});
    }
}