/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.guice;

import java.lang.reflect.Method;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.WordUtils;
import org.cattleframework.aop.annotation.GuiceBindBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.core.annotation.MergedAnnotation;
import org.springframework.core.annotation.MergedAnnotations;
import org.springframework.core.annotation.MergedAnnotations.SearchStrategy;

import com.google.inject.AbstractModule;
import com.google.inject.Injector;
import com.google.inject.matcher.Matchers;
import com.google.inject.name.Names;
import com.google.inject.spi.ProvisionListener;

/**
 * Spring的Guice模块
 * 
 * @author orange
 *
 */
public class SpringModule extends AbstractModule {

    private static final Logger logger = LoggerFactory.getLogger(SpringModule.class);

    public static final String CATTLE_GUICE_SOURCE = "cattle-guice";

    private ConfigurableListableBeanFactory beanFactory;

    public SpringModule(ConfigurableListableBeanFactory beanFactory) {
	this.beanFactory = beanFactory;
    }

    @Override
    protected void configure() {
	if (beanFactory.getBeanNamesForType(ProvisionListener.class).length > 0) {
	    bindListener(Matchers.any(),
		    beanFactory.getBeansOfType(ProvisionListener.class).values().toArray(new ProvisionListener[0]));
	}
	if (beanFactory instanceof DefaultListableBeanFactory) {
	    ((DefaultListableBeanFactory) beanFactory).setAutowireCandidateResolver(
		    new GuiceAutowireCandidateResolver(binder().getProvider(Injector.class)));
	}
	bind();
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private void bind() {
	for (String name : beanFactory.getBeanDefinitionNames()) {
	    BeanDefinition definition = beanFactory.getBeanDefinition(name);
	    if (definition.hasAttribute(CATTLE_GUICE_SOURCE)) {
		continue;
	    }
	    if (definition.isAutowireCandidate() && definition.getRole() == AbstractBeanDefinition.ROLE_APPLICATION) {
		Class<?> clazz = beanFactory.getType(name);
		if (clazz == null) {
		    continue;
		}
		String sName = WordUtils.uncapitalize(clazz.getSimpleName());
		GuiceBindBean guiceBindBean = clazz.getAnnotation(GuiceBindBean.class);
		if (guiceBindBean != null) {
		    MergedAnnotation<ConfigurationProperties> annotation = MergedAnnotations
			    .from(clazz, SearchStrategy.TYPE_HIERARCHY).get(ConfigurationProperties.class);
		    if (!"org.springframework.core.annotation.MissingMergedAnnotation"
			    .equals(annotation.getClass().getName())) {
			String prefix = annotation.isPresent() ? annotation.getString("prefix") : "";
			sName = StringUtils.isNotBlank(prefix) ? prefix + "-" + clazz.getName() : clazz.getName();
		    }
		    SpringBeanProvider beanFactoryProvider = new SpringBeanProvider(beanFactory, name, clazz);
		    if (!name.equals(sName)) {
			bind(clazz).annotatedWith(Names.named(name)).toProvider(beanFactoryProvider);
			logger.debug("注入Guice,实现类:{},别名:{}", clazz.getName(), name);
		    } else {
			bind(clazz).toProvider(beanFactoryProvider);
			logger.debug("注入Guice,实现类:{}", clazz.getName());
		    }
		} else {
		    String factoryBeanName = definition.getFactoryBeanName();
		    String factoryMethodName = definition.getFactoryMethodName();
		    if (StringUtils.isNotBlank(factoryBeanName) && StringUtils.isNotBlank(factoryMethodName)) {
			if (isBind(factoryBeanName, factoryMethodName, clazz)) {
			    SpringBeanProvider beanFactoryProvider = new SpringBeanProvider(beanFactory, name, clazz);
			    if (!name.equals(sName)) {
				bind(clazz).annotatedWith(Names.named(name)).toProvider(beanFactoryProvider);
				logger.debug("注入Guice,实现类:{},别名:{}", clazz.getName(), name);
			    } else {
				bind(clazz).toProvider(beanFactoryProvider);
				logger.debug("注入Guice,实现类:{}", clazz.getName());
			    }
			}
		    }
		}
	    }
	}
    }

    private boolean isBind(String factoryBeanName, String factoryMethodName, Class<?> returnType) {
	boolean bind = false;
	Class<?> factoryClass = beanFactory.getType(factoryBeanName);
	for (Method method : factoryClass.getDeclaredMethods()) {
	    if (method.getName().equals(factoryMethodName) && method.getAnnotation(Bean.class) != null
		    && method.getAnnotation(GuiceBindBean.class) != null && method.getReturnType() == returnType) {
		bind = true;
		break;
	    }
	}
	return bind;
    }
}