/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.initialize;

import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.cattleframework.aop.event.ContextInitializeEvent;
import org.cattleframework.aop.utils.ServiceLoaderUtils;
import org.cattleframework.exception.CommonException;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;

/**
 * 上下文初始化
 * 
 * @author orange
 *
 */
public class ContextInitializer implements ApplicationContextInitializer<ConfigurableApplicationContext> {

    private static final Logger logger = LoggerFactory.getLogger(ContextInitializer.class);

    @Override
    public void initialize(ConfigurableApplicationContext applicationContext) {
	ConfigurableListableBeanFactory beanFactory = applicationContext.getBeanFactory();
	ConfigurableEnvironment environment = applicationContext.getEnvironment();
	Set<ContextInitializeEvent> contextInitializeEvents;
	try {
	    contextInitializeEvents = ServiceLoaderUtils.getService(ContextInitializeEvent.class);
	} catch (CommonException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
	if (CollectionUtils.isNotEmpty(contextInitializeEvents)) {
	    processInitializeEvent(beanFactory, environment, contextInitializeEvents);
	}
    }

    private void processInitializeEvent(ConfigurableListableBeanFactory beanFactory,
	    ConfigurableEnvironment environment, Set<ContextInitializeEvent> contextInitializeEvents) {
	contextInitializeEvents.stream().forEachOrdered(contextInitializeEvent -> {
	    try {
		logger.debug("处理上下文初始化事件,类名:{}", contextInitializeEvent.getClass().getName());
		contextInitializeEvent.execute(beanFactory, environment);
	    } catch (Throwable e) {
		throw ExceptionWrapUtils.wrapRuntime(e);
	    }
	});
    }
}