package org.cattleframework.aop.proxy;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.aopalliance.intercept.MethodInterceptor;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.constants.ProxyConstants;
import org.cattleframework.aop.constants.ProxyMode;
import org.cattleframework.aop.constants.ScanMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.TargetSource;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.framework.autoproxy.AbstractAutoProxyCreator;
import org.springframework.beans.BeansException;
import org.springframework.stereotype.Component;

/**
 * 自动扫描代理抽象
 * 
 * @author orange
 *
 */
public abstract class AbstractAutoScanProxy extends AbstractAutoProxyCreator {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory.getLogger(AbstractAutoScanProxy.class);

    private static final String JAVA_PACKAGE_PREFIX = "java.";

    private static final String SEPARATOR = ";";

    /**
     * 获取所有附加拦截器
     * 
     * @param targetClass 目标类
     * @return 所有附加拦截器
     */
    protected abstract MethodInterceptor[] getAdditionalInterceptors(Class<?> targetClass);

    /**
     * 获取所有通用拦截器
     * 
     * @return 所有通用拦截器
     */
    protected abstract Class<? extends MethodInterceptor>[] getCommonInterceptors();

    /**
     * 获取所有通用拦截器名称
     * 
     * @return 所有通用拦截器名称
     */
    protected abstract String[] getCommonInterceptorNames();

    /**
     * 获取所有方法注解
     * 
     * @return 所有方法注解
     */
    protected abstract Class<? extends Annotation>[] getMethodAnnotations();

    /**
     * 方法注解已扫描
     * 
     * @param targetClass      目标类
     * @param method           方法
     * @param methodAnnotation 方法注解
     */
    protected abstract void methodAnnotationScanned(Class<?> targetClass, Method method,
	    Class<? extends Annotation> methodAnnotation);

    /**
     * 获取所有类注解
     * 
     * @return 所有类注解
     */
    protected abstract Class<? extends Annotation>[] getClassAnnotations();

    /**
     * 类注解已扫描
     * 
     * @param targetClass     目标类
     * @param classAnnotation 类注解
     */
    protected abstract void classAnnotationScanned(Class<?> targetClass, Class<? extends Annotation> classAnnotation);

    private String[] scanPackages;

    private ProxyMode proxyMode;

    private ScanMode scanMode;

    private final Map<String, Object> beanMap = new HashMap<String, Object>();

    private final Map<String, Boolean> proxyMap = new HashMap<String, Boolean>();

    private final Map<String, Boolean> proxyTargetClassMap = new HashMap<String, Boolean>();

    public AbstractAutoScanProxy(String[] scanPackages, ProxyMode proxyMode, ScanMode scanMode) {
	this(scanPackages, proxyMode, scanMode, true);
    }

    public AbstractAutoScanProxy(String[] scanPackages, ProxyMode proxyMode, ScanMode scanMode, boolean exposeProxy) {
	this.scanPackages = scanPackages;
	this.setExposeProxy(exposeProxy);
	this.proxyMode = proxyMode;
	this.scanMode = scanMode;
	StringBuilder builder = new StringBuilder();
	if (ArrayUtils.isNotEmpty(scanPackages)) {
	    for (int i = 0; i < scanPackages.length; i++) {
		String scanPackage = scanPackages[i];
		builder.append(scanPackage);
		if (i < scanPackages.length - 1) {
		    builder.append(SEPARATOR);
		}
	    }
	}
	logger.debug("------------- Matrix Aop Information ------------");
	logger.debug("Auto scan proxy class is {}", getClass().getCanonicalName());
	logger.debug("Scan packages is {}", ArrayUtils.isNotEmpty(scanPackages) ? builder.toString() : "not set");
	logger.debug("Proxy mode is {}", proxyMode);
	logger.debug("Scan mode is {}", scanMode);
	logger.debug("Expose proxy is {}", exposeProxy);
	logger.debug("-------------------------------------------------");
	Class<? extends MethodInterceptor>[] commonInterceptorClasses = getCommonInterceptors();
	String[] commonInterceptorNames = getCommonInterceptorNames();
	String[] interceptorNames = ArrayUtils.addAll(commonInterceptorNames,
		convertInterceptorNames(commonInterceptorClasses));
	if (ArrayUtils.isNotEmpty(interceptorNames)) {
	    setInterceptorNames(interceptorNames);
	}
    }

    @Override
    protected Object[] getAdvicesAndAdvisorsForBean(Class<?> beanClass, String beanName,
	    TargetSource customTargetSource) throws BeansException {
	if (ArrayUtils.isNotEmpty(scanPackages)) {
	    boolean scanPackagesContained = scanPackagesContained(beanClass);
	    if (!scanPackagesContained) {
		return DO_NOT_PROXY;
	    }
	}
	Object bean = beanMap.get(beanName);
	Class<?> targetClass = null;
	if (bean != null) {
	    targetClass = AopProxyUtils.ultimateTargetClass(bean);
	} else {
	    targetClass = beanClass;
	}
	if (!targetClass.isInterface()) {
	    if (targetClass.getInterfaces() != null) {
		for (Class<?> targetInterface : targetClass.getInterfaces()) {
		    Object[] proxyInterceptors = scanAndProxyForTarget(targetInterface, beanName, false);
		    if (proxyInterceptors != DO_NOT_PROXY) {
			return proxyInterceptors;
		    }
		}
	    }
	    Object[] proxyInterceptors = scanAndProxyForTarget(targetClass, beanName, true);
	    if (proxyInterceptors != DO_NOT_PROXY) {
		return proxyInterceptors;
	    }
	}
	return DO_NOT_PROXY;
    }

    private boolean scanPackagesContained(Class<?> beanClass) {
	for (String scanPackage : scanPackages) {
	    if (StringUtils.isNotBlank(scanPackage)) {
		String beanClassName = beanClass.getCanonicalName();
		if (StringUtils.isNotBlank(beanClassName)) {
		    if (beanClassName.startsWith(scanPackage)
			    || beanClassName.contains(ProxyConstants.JDK_PROXY_NAME_KEY)
			    || beanClassName.contains(ProxyConstants.CGLIB_PROXY_NAME_KEY)) {
			return true;
		    }
		} else {
		    return false;
		}
	    }
	}
	return false;
    }

    private Object[] scanAndProxyForTarget(Class<?> targetClass, String beanName, boolean proxyTargetClass) {
	String targetClassName = targetClass.getCanonicalName();
	Object[] interceptors = getInterceptors(targetClass);
	if (StringUtils.isNotBlank(targetClassName) && !targetClassName.startsWith(JAVA_PACKAGE_PREFIX)) {
	    Boolean proxied = proxyMap.get(targetClassName);
	    if (proxied != null) {
		if (proxied) {
		    return interceptors;
		}
	    } else {
		Object[] proxyInterceptors = null;
		if (proxyMode == ProxyMode.BY_CLASS_ANNOTATION_ONLY) {
		    proxyInterceptors = scanAndProxyForClass(targetClass, targetClassName, beanName, interceptors,
			    proxyTargetClass);
		} else if (proxyMode == ProxyMode.BY_METHOD_ANNOTATION_ONLY) {
		    proxyInterceptors = scanAndProxyForMethod(targetClass, targetClassName, beanName, interceptors,
			    proxyTargetClass);
		} else if (proxyMode == ProxyMode.BY_CLASS_OR_METHOD_ANNOTATION) {
		    Object[] classProxyInterceptors = scanAndProxyForClass(targetClass, targetClassName, beanName,
			    interceptors, proxyTargetClass);
		    Object[] methodProxyInterceptors = scanAndProxyForMethod(targetClass, targetClassName, beanName,
			    interceptors, proxyTargetClass);
		    if (classProxyInterceptors != DO_NOT_PROXY || methodProxyInterceptors != DO_NOT_PROXY) {
			proxyInterceptors = interceptors;
		    } else {
			proxyInterceptors = DO_NOT_PROXY;
		    }
		}
		proxyMap.put(targetClassName, Boolean.valueOf(proxyInterceptors != DO_NOT_PROXY));
		if (proxyInterceptors != DO_NOT_PROXY) {
		    proxyTargetClassMap.put(beanName, proxyTargetClass);
		}
		return proxyInterceptors;
	    }
	}
	return DO_NOT_PROXY;
    }

    private Object[] getInterceptors(Class<?> targetClass) {
	Object[] interceptors = getAdditionalInterceptors(targetClass);
	if (ArrayUtils.isNotEmpty(interceptors)) {
	    return interceptors;
	}
	Class<? extends MethodInterceptor>[] commonInterceptorClasses = getCommonInterceptors();
	String[] commonInterceptorNames = getCommonInterceptorNames();
	if (ArrayUtils.isNotEmpty(commonInterceptorClasses) || ArrayUtils.isNotEmpty(commonInterceptorNames)) {
	    return PROXY_WITHOUT_ADDITIONAL_INTERCEPTORS;
	}
	return DO_NOT_PROXY;
    }

    private Object[] scanAndProxyForMethod(Class<?> targetClass, String targetClassName, String beanName,
	    Object[] interceptors, boolean proxyTargetClass) {
	boolean proxied = false;
	Class<? extends Annotation>[] methodAnnotations = getMethodAnnotations();
	if (ArrayUtils.isNotEmpty(methodAnnotations)) {
	    for (Method method : targetClass.getDeclaredMethods()) {
		for (Class<? extends Annotation> methodAnnotation : methodAnnotations) {
		    if (method.isAnnotationPresent(methodAnnotation)) {
			if (scanMode == ScanMode.FOR_METHOD_ANNOTATION_ONLY
				|| scanMode == ScanMode.FOR_CLASS_OR_METHOD_ANNOTATION) {
			    methodAnnotationScanned(targetClass, method, methodAnnotation);
			} else {
			    return interceptors;
			}
			if (!proxied) {
			    proxied = true;
			}
		    }
		}
	    }
	}
	return proxied ? interceptors : DO_NOT_PROXY;
    }

    private Object[] scanAndProxyForClass(Class<?> targetClass, String targetClassName, String beanName,
	    Object[] interceptors, boolean proxyTargetClass) {
	boolean proxied = false;
	Class<? extends Annotation>[] classAnnotations = getClassAnnotations();
	if (ArrayUtils.isNotEmpty(classAnnotations)) {
	    for (Class<? extends Annotation> classAnnotation : classAnnotations) {
		if (targetClass.isAnnotationPresent(classAnnotation)) {
		    if (scanMode == ScanMode.FOR_CLASS_ANNOTATION_ONLY
			    || scanMode == ScanMode.FOR_CLASS_OR_METHOD_ANNOTATION) {
			classAnnotationScanned(targetClass, classAnnotation);
		    } else {
			return interceptors;
		    }
		    if (!proxied) {
			proxied = true;
		    }
		}
	    }
	}
	return proxied ? interceptors : DO_NOT_PROXY;
    }

    private String[] convertInterceptorNames(Class<? extends MethodInterceptor>[] commonInterceptorClasses) {
	if (ArrayUtils.isNotEmpty(commonInterceptorClasses)) {
	    String[] interceptorNames = new String[commonInterceptorClasses.length];
	    for (int i = 0; i < commonInterceptorClasses.length; i++) {
		Class<? extends MethodInterceptor> interceptorClass = commonInterceptorClasses[i];
		interceptorNames[i] = interceptorClass.getAnnotation(Component.class).value();
	    }
	    return interceptorNames;
	}
	return null;
    }

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
	Object object = super.postProcessBeforeInitialization(bean, beanName);
	if (ArrayUtils.isNotEmpty(scanPackages)) {
	    boolean scanPackagesContained = scanPackagesContained(bean.getClass());
	    if (scanPackagesContained) {
		beanMap.put(beanName, bean);
	    }
	} else {
	    beanMap.put(beanName, bean);
	}
	return object;
    }

    @Override
    protected boolean shouldProxyTargetClass(Class<?> beanClass, String beanName) {
	Boolean proxyTargetClass = proxyTargetClassMap.get(beanName);
	if (proxyTargetClass != null) {
	    return proxyTargetClass;
	}
	return super.shouldProxyTargetClass(beanClass, beanName);
    }
}