package org.cattleframework.aop.proxy;

import java.lang.reflect.Method;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.cattleframework.aop.annotation.IgnoreAutoScanProxy;
import org.springframework.core.ParameterNameDiscoverer;
import org.springframework.core.StandardReflectionParameterNameDiscoverer;
import org.springframework.web.bind.annotation.InitBinder;

/**
 * 拦截抽象
 * 
 * @author orange
 *
 */
public abstract class AbstractInterceptor implements MethodInterceptor {

    private ParameterNameDiscoverer standardReflectionParameterNameDiscoverer = new StandardReflectionParameterNameDiscoverer();

    /**
     * 调用拦截器
     * 
     * @param invocation 方法调用
     * @return 调用方法后的结果
     * @throws Throwable 异常
     */
    @Nullable
    public abstract Object invokeInterceptor(@Nonnull MethodInvocation invocation) throws Throwable;

    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
	boolean hasInitBinderAnnotation = getMethod(invocation).isAnnotationPresent(InitBinder.class);
	if (hasInitBinderAnnotation) {
	    return invocation.proceed();
	}
	boolean hasIgnoreAutoScanProxy = getMethod(invocation).isAnnotationPresent(IgnoreAutoScanProxy.class)
		|| (invocation.getThis() != null
			&& invocation.getThis().getClass().isAnnotationPresent(IgnoreAutoScanProxy.class));
	if (hasIgnoreAutoScanProxy) {
	    return invocation.proceed();
	}
	return invokeInterceptor(invocation);
    }

    public Method getMethod(MethodInvocation invocation) {
	return invocation.getMethod();
    }

    public String getMethodName(MethodInvocation invocation) {
	return getMethod(invocation).getName();
    }

    public String[] getMethodParameterNames(MethodInvocation invocation) {
	Method method = getMethod(invocation);
	return standardReflectionParameterNameDiscoverer.getParameterNames(method);
    }

    public Object[] getArguments(MethodInvocation invocation) {
	return invocation.getArguments();
    }

    public String getProxyClassName(MethodInvocation invocation) {
	return getProxyClass(invocation).getCanonicalName();
    }

    public Class<?> getProxyClass(MethodInvocation invocation) {
	return invocation.getClass();
    }
}