/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.reflections;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.annotation.ClassLoadLevel;
import org.cattleframework.aop.annotation.ConditionalOnClass;
import org.cattleframework.aop.utils.SimpleReflectUtils;
import org.reflections.Reflections;
import org.reflections.util.ConfigurationBuilder;

/**
 * Reflections工厂
 * 
 * @author orange
 *
 */
public class ReflectionsFactory {

    private static String[] scanPackages;

    private static Reflections reflections;

    public static void initialize(String[] scanPackages) {
	ReflectionsFactory.scanPackages = scanPackages;
	ConfigurationBuilder configurationBuilder = new ConfigurationBuilder();
	configurationBuilder = configurationBuilder.forPackages(scanPackages);
	reflections = new Reflections(configurationBuilder);
    }

    public static String[] getScanPackages() {
	return scanPackages;
    }

    public static Set<Class<?>> getTypesAnnotatedWith(Class<? extends Annotation> annotation) {
	Set<Class<?>> classes = reflections.getTypesAnnotatedWith(annotation);
	Set<Class<?>> rClasses = new HashSet<Class<?>>();
	classes.forEach(item -> {
	    if (SimpleReflectUtils.isClass(item)) {
		Class<?> clazz = checkConditionalOnClass(item);
		if (clazz != null) {
		    rClasses.add(clazz);
		}
	    }
	});
	return rClasses;
    }

    public static <T> Class<? extends T> getSubTypesOfByLoadLevel(Class<T> type) {
	Set<Class<? extends T>> classes = getSubTypesOf(type);
	Class<? extends T> result = null;
	int currentLevel = -1;
	for (Class<? extends T> lClazz : classes) {
	    ClassLoadLevel loadLevel = lClazz.getAnnotation(ClassLoadLevel.class);
	    int level = loadLevel != null ? loadLevel.value() : 0;
	    if (level > currentLevel) {
		result = lClazz;
		currentLevel = level;
	    }
	}
	return result;
    }

    @SuppressWarnings("unchecked")
    public static <T> Set<Class<? extends T>> getSubTypesOf(Class<T> type) {
	Set<Class<? extends T>> classes = reflections.getSubTypesOf(type);
	Set<Class<? extends T>> rClasses = new HashSet<Class<? extends T>>();
	classes.stream().forEach(item -> {
	    if (SimpleReflectUtils.isClass(item)) {
		Class<? extends T> clazz = (Class<? extends T>) checkConditionalOnClass(item);
		if (clazz != null) {
		    rClasses.add((Class<T>) clazz);
		}
	    }
	});
	return rClasses;
    }

    private static Class<?> checkConditionalOnClass(Class<?> clazz) {
	ConditionalOnClass conditionalOnClass = clazz.getAnnotation(ConditionalOnClass.class);
	if (conditionalOnClass != null) {
	    String[] classNames = conditionalOnClass.value();
	    if (classNames != null && classNames.length > 0) {
		for (int i = 0; i < classNames.length; i++) {
		    if (StringUtils.isNotBlank(classNames[i])) {
			if (!SimpleReflectUtils.classExists(classNames[i])) {
			    return null;
			}
		    }
		}
	    }
	}
	return clazz;
    }
}