/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.resource.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.resource.ClassResourceLoader;
import org.cattleframework.aop.resource.Resource;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.ResourcePatternUtils;

/**
 * Spring类资源加载器
 * 
 * @author orange
 *
 */
public class ClassResourceLoaderImpl implements ClassResourceLoader {

    private static final String SEPARATE = "/";

    private static final String POINT = ".";

    private static final String DEFAULT_FILE_TAG = "-default";

    private final ResourceLoader resourceLoader;

    public ClassResourceLoaderImpl(ResourceLoader resourceLoader) {
	this.resourceLoader = resourceLoader;
    }

    @Override
    public Resource getResource(String location) {
	if (StringUtils.isBlank(location)) {
	    return null;
	}
	org.springframework.core.io.Resource resource = resourceLoader.getResource(location);
	if (resource != null && resource.exists()) {
	    return new ResourceImpl(resource);
	}
	return null;
    }

    @Override
    public Resource getResource(String[] locations, String resourceName) {
	return getResource(locations, resourceName, false);
    }

    @Override
    public Resource getResource(String[] locations, String resourceName, boolean allowDefault) {
	Resource resource = null;
	if (ArrayUtils.isNotEmpty(locations)) {
	    for (String location : locations) {
		if (StringUtils.isBlank(location)) {
		    continue;
		}
		resource = getResource(getResourceLocation(location, resourceName, false));
		if (resource != null) {
		    break;
		}
		if (allowDefault && resource == null) {
		    resource = getResource(getResourceLocation(location, resourceName, true));
		    if (resource != null) {
			break;
		    }
		}
	    }
	}
	return resource;
    }

    @Override
    public Set<Resource> getResources(String locationPattern) {
	try {
	    org.springframework.core.io.Resource[] resources = ResourcePatternUtils
		    .getResourcePatternResolver(resourceLoader).getResources(locationPattern);
	    Set<Resource> result = new LinkedHashSet<Resource>();
	    Arrays.stream(resources).forEach(resource -> {
		if (resource.exists()) {
		    result.add(new ResourceImpl(resource));
		}
	    });
	    return result;
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public Set<Resource> getResources(String locationPattern, String suffix) {
	if (StringUtils.isBlank(locationPattern)) {
	    return null;
	}
	try {
	    org.springframework.core.io.Resource[] resources = ResourcePatternUtils
		    .getResourcePatternResolver(resourceLoader).getResources(locationPattern);
	    Set<Resource> result = new LinkedHashSet<Resource>();
	    Arrays.stream(resources).forEach(resource -> {
		if (resource.exists()) {
		    String path;
		    try {
			path = resource.getURL().getPath();
		    } catch (IOException e) {
			throw ExceptionWrapUtils.wrapRuntime(e);
		    }
		    boolean hasSuffix = path.indexOf('.') >= 0;
		    boolean exist = (hasSuffix && StringUtils.isNotBlank(suffix) && path.endsWith("." + suffix))
			    || (!hasSuffix && StringUtils.isBlank(suffix));
		    if (exist) {
			result.add(new ResourceImpl(resource));
		    }
		}
	    });
	    return result;
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    private String getResourceLocation(String location, String resourceName, boolean isDefault) {
	String resourceLocation = location.endsWith(SEPARATE) ? location.substring(0, location.length() - 1) : location;
	String lResourceName = resourceName;
	if (!lResourceName.startsWith(SEPARATE)) {
	    lResourceName = SEPARATE + resourceName;
	}
	if (isDefault) {
	    int pos = lResourceName.lastIndexOf(POINT);
	    if (pos >= 0) {
		lResourceName = lResourceName.substring(0, pos) + DEFAULT_FILE_TAG + lResourceName.substring(pos);
	    } else {
		lResourceName += DEFAULT_FILE_TAG;
	    }
	}
	resourceLocation += lResourceName;
	return resourceLocation;
    }
}