/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.resource.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;

import org.cattleframework.aop.resource.Resource;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.springframework.util.ResourceUtils;

/**
 * Spring资源实现
 * 
 * @author orange
 *
 */
public class ResourceImpl implements Resource {

    private static final String NESTED_PREFIX = "nested:";

    private org.springframework.core.io.Resource resource;

    private String description;

    public ResourceImpl(org.springframework.core.io.Resource resource) {
	this.resource = resource;
	if (resource != null) {
	    try {
		String file = resource.getURL().getFile();
		if (file.startsWith(ResourceUtils.FILE_URL_PREFIX)) {
		    file = file.substring(ResourceUtils.FILE_URL_PREFIX.length());
		}
		if (file.startsWith(NESTED_PREFIX)) {
		    file = file.substring(NESTED_PREFIX.length());
		}
		description = file;
	    } catch (IOException e) {
	    }
	}
    }

    @Override
    public URL getUrl() {
	try {
	    return resource.getURL();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public String getDescription() {
	return description;
    }

    @Override
    public InputStream getInputStream() {
	try {
	    return resource.getInputStream();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public URI getUri() {
	try {
	    return resource.getURI();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public File getFile() {
	try {
	    return resource.getFile();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public long lastModified() {
	try {
	    return resource.lastModified();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public long contentLength() {
	try {
	    return resource.contentLength();
	} catch (IOException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
    }

    @Override
    public boolean isFile() {
	return resource.isReadable();
    }

    @Override
    public String getFileName() {
	return resource.getFilename();
    }

    @Override
    public String getPath() {
	String path = getUrl().getPath();
	int index = path.lastIndexOf("!/");
	return index >= 0 ? path.substring(index + 2) : getFile().getPath();
    }
}