/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.selector;

import java.util.List;

import org.springframework.beans.factory.BeanClassLoaderAware;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.DeferredImportSelector;
import org.springframework.core.GenericTypeResolver;
import org.springframework.core.env.Environment;
import org.springframework.core.io.support.SpringFactoriesLoader;
import org.springframework.core.type.AnnotationMetadata;

/**
 * 导入选择器抽象
 * 
 * @author orange
 *
 * @param <T>
 */
public abstract class AbstractImportSelector<T>
	implements DeferredImportSelector, BeanClassLoaderAware, EnvironmentAware {

    private ClassLoader beanClassLoader;

    private Class<T> annotationClass;

    private Environment environment;

    /**
     * 是否启用
     * 
     * @return 是否启用
     */
    protected abstract boolean isEnabled();

    @SuppressWarnings("unchecked")
    protected AbstractImportSelector() {
	this.annotationClass = (Class<T>) GenericTypeResolver.resolveTypeArgument(getClass(),
		AbstractImportSelector.class);
    }

    @SuppressWarnings("deprecation")
    @Override
    public String[] selectImports(AnnotationMetadata importingClassMetadata) {
	if (!isEnabled()) {
	    return new String[0];
	}
	List<String> factoryNames = SpringFactoriesLoader.loadFactoryNames(annotationClass, beanClassLoader);
	return factoryNames.toArray(new String[0]);
    }

    @Override
    public void setEnvironment(Environment environment) {
	this.environment = environment;
    }

    @Override
    public void setBeanClassLoader(ClassLoader classLoader) {
	beanClassLoader = classLoader;
    }

    public Environment getEnvironment() {
	return environment;
    }
}