/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.aop.utils;

import java.util.HashSet;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.annotation.ClassLoadLevel;
import org.cattleframework.aop.annotation.ConditionalOnClass;
import org.cattleframework.exception.CommonException;

/**
 * 服务加载器工具
 * 
 * @author orange
 *
 */
public class ServiceLoaderUtils {

    private static ConcurrentMap<String, ServiceLoader<?>> serviceLoaderCache = new ConcurrentHashMap<String, ServiceLoader<?>>();

    public static <T> T getServiceByLoadLevel(Class<T> service) throws CommonException {
	return getServiceByLoadLevel(service, false);
    }

    public static <T> T getServiceByLoadLevel(Class<T> service, boolean cache) throws CommonException {
	ServiceLoader<T> serviceLoader = getServiceLoader(service, cache);
	T result = null;
	int currentLevel = -1;
	for (T lService : serviceLoader) {
	    if (checkConditionalOnClass(lService.getClass())) {
		ClassLoadLevel loadLevel = lService.getClass().getAnnotation(ClassLoadLevel.class);
		int level = loadLevel != null ? loadLevel.value() : 0;
		if (level > currentLevel) {
		    result = lService;
		    currentLevel = level;
		}
	    }
	}
	return result;
    }

    public static <T> Set<T> getService(Class<T> service) throws CommonException {
	return getService(service, false);
    }

    public static <T> Set<T> getService(Class<T> service, boolean cache) throws CommonException {
	ServiceLoader<T> serviceLoader = getServiceLoader(service, cache);
	Set<T> result = new HashSet<T>();
	for (T lService : serviceLoader) {
	    if (checkConditionalOnClass(lService.getClass())) {
		result.add(lService);
	    }
	}
	return result;
    }

    private static boolean checkConditionalOnClass(Class<?> clazz) {
	ConditionalOnClass conditionalOnClass = clazz.getAnnotation(ConditionalOnClass.class);
	if (conditionalOnClass != null) {
	    String[] classNames = conditionalOnClass.value();
	    if (classNames != null && classNames.length > 0) {
		for (int i = 0; i < classNames.length; i++) {
		    if (StringUtils.isNotBlank(classNames[i])) {
			if (!SimpleReflectUtils.classExists(classNames[i])) {
			    return false;
			}
		    }
		}
	    }
	}
	return true;
    }

    @SuppressWarnings("unchecked")
    private static <T> ServiceLoader<T> getServiceLoader(Class<T> service, boolean cache) throws CommonException {
	if (!SimpleReflectUtils.isPublicClass(service) && !SimpleReflectUtils.isInterfaceClass(service)) {
	    throw new CommonException("'" + service.getName() + "'不是公共接口类");
	}
	ServiceLoader<T> serviceLoader = null;
	String serviceName = service.getName();
	if (cache) {
	    if (serviceLoaderCache.containsKey(serviceName)) {
		serviceLoader = (ServiceLoader<T>) serviceLoaderCache.get(serviceName);
	    }
	}
	if (serviceLoader == null) {
	    serviceLoader = ServiceLoader.load(service);
	    if (cache) {
		synchronized (serviceLoaderCache) {
		    serviceLoaderCache.put(serviceName, serviceLoader);
		}
	    }
	}
	return serviceLoader;
    }
}