/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.cloud.config;

import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.event.ContextInitializeEvent;
import org.cattleframework.aop.processor.SortedBeanPostProcessor;
import org.cattleframework.aop.utils.ServiceLoaderUtils;
import org.cattleframework.cloud.config.processor.ConfigPropertiesProcessor;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.core.Ordered;
import org.springframework.core.env.Environment;

/**
 * 服务配置初始化
 * 
 * @author orange
 *
 */
public class ConfigInitialize implements ContextInitializeEvent {

    @Override
    public void execute(ConfigurableListableBeanFactory beanFactory, Environment environment) throws Throwable {
	List<ConfigPropertiesProcessor> configPropertiesProcessors = ServiceLoaderUtils
		.getService(ConfigPropertiesProcessor.class);
	String processorName = StringUtils.uncapitalize(getClass().getSimpleName()) + "Processor";
	beanFactory.registerSingleton(processorName, new SortedBeanPostProcessor() {

	    @Override
	    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
		List<ConfigPropertiesProcessor> configPropertiesProcessorLst = configPropertiesProcessors.stream()
			.filter(p -> p.needProcess(bean.getClass())).collect(Collectors.toList());
		if (CollectionUtils.isNotEmpty(configPropertiesProcessorLst)) {
		    configPropertiesProcessorLst.forEach(
			    configPropertiesProcessor -> configPropertiesProcessor.process(beanFactory, environment,
				    bean, beanFactory.getBean(configPropertiesProcessor.getSourcePropertiesClass())));
		}
		return bean;
	    }

	    @Override
	    public int getOrder() {
		return Ordered.HIGHEST_PRECEDENCE;
	    }
	});
    }
}