/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.cloud.config.rule;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.aop.context.SpringContext;
import org.cattleframework.aop.utils.SimpleReflectUtils;
import org.cattleframework.cloud.config.rule.processor.RuleBeanInfo;
import org.cattleframework.cloud.config.rule.processor.RuleConfigResourceProcessor;
import org.cattleframework.cloud.config.rule.processor.RuleProcessor;
import org.cattleframework.cloud.config.rule.property.RulePropertyDelegate;
import org.cattleframework.cloud.config.rule.resource.BaseRuleConfigResource;
import org.cattleframework.exception.CommonException;
import org.cattleframework.exception.CommonRuntimeException;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.core.env.Environment;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * 规则配置资源初始化
 * 
 * @author orange
 *
 */
public class RuleConfigResourceInitializer implements SmartInitializingSingleton {

    private static final Logger logger = LoggerFactory.getLogger(RuleConfigResourceInitializer.class);

    private static final String RULE_SUFFIX = "Rule";

    private final DefaultListableBeanFactory beanFactory;

    private final Environment environment;

    private final RuleConfigProperties ruleConfigProperties;

    private final List<RuleConfigResourceProcessor> ruleConfigResourceProcessors;

    private final List<RuleProcessor> ruleProcessors;

    public RuleConfigResourceInitializer(DefaultListableBeanFactory beanFactory, Environment environment,
	    RuleConfigProperties ruleConfigProperties, List<RuleConfigResourceProcessor> ruleConfigResourceProcessors,
	    List<RuleProcessor> ruleProcessors) {
	this.beanFactory = beanFactory;
	this.environment = environment;
	this.ruleConfigProperties = ruleConfigProperties;
	this.ruleConfigResourceProcessors = ruleConfigResourceProcessors;
	this.ruleProcessors = ruleProcessors;
    }

    @Override
    public void afterSingletonsInstantiated() {
	if (CollectionUtils.isEmpty(ruleConfigResourceProcessors) || CollectionUtils.isEmpty(ruleProcessors)) {
	    return;
	}
	if (StringUtils.isBlank(SpringContext.get().getApplicationName())) {
	    throw new CommonRuntimeException("没有设置应用服务名");
	}
	String fileExtension = ruleConfigProperties.getFileExtension();
	if (StringUtils.isBlank(fileExtension)) {
	    throw new CommonRuntimeException("规则文件类型为空");
	}
	if (!RuleConfigConstants.RULE_FILE_EXT_JSON.equalsIgnoreCase(fileExtension)
		&& !RuleConfigConstants.RULE_FILE_EXT_XML.equals(fileExtension)) {
	    throw new CommonRuntimeException(String.format("规则文件类型'%s'不支持", fileExtension));
	}
	String mapperClassName = null;
	if (RuleConfigConstants.RULE_FILE_EXT_JSON.equalsIgnoreCase(fileExtension)) {
	    mapperClassName = "com.fasterxml.jackson.databind.ObjectMapper";
	} else if (RuleConfigConstants.RULE_FILE_EXT_XML.equals(fileExtension)) {
	    mapperClassName = "com.fasterxml.jackson.dataformat.xml.XmlMapper";
	}
	if (StringUtils.isBlank(mapperClassName)) {
	    throw new CommonRuntimeException(String.format("规则文件类型'%s'没有相应规则转换器工厂Bean名称", fileExtension));
	}
	Class<?> mapperClass;
	try {
	    mapperClass = ClassUtils.getClass(mapperClassName, true);
	} catch (ClassNotFoundException e) {
	    throw ExceptionWrapUtils.wrapRuntime(e);
	}
	if (mapperClass == null) {
	    throw new CommonRuntimeException(String.format("规则文件类型'%s'没有找到相应序列化类'%s'", fileExtension, mapperClassName));
	}
	ObjectMapper objectMapper;
	try {
	    objectMapper = (ObjectMapper) SimpleReflectUtils.instance(mapperClass);
	} catch (CommonException e) {
	    throw new CommonRuntimeException(
		    String.format("规则文件类型'%s'的相应序列化类'%s',初始化异常:%s", fileExtension, mapperClassName, e.getMessage()), e);
	}
	objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
	ruleConfigResourceProcessors.forEach(ruleConfigResourceProcessor -> {
	    ruleProcessors.forEach(ruleProcessor -> {
		RuleBeanInfo<?>[] beanInfos = ruleProcessor.getBeanInfos();
		Arrays.stream(beanInfos).forEach(beanInfo -> {
		    String ruleName = org.cattleframework.utils.auxiliary.StringUtils
			    .camel2Hyphen(beanInfo.getRuleName().endsWith(RULE_SUFFIX)
				    ? beanInfo.getRuleName().substring(0,
					    beanInfo.getRuleName().length() - RULE_SUFFIX.length())
				    : beanInfo.getRuleName());
		    try {
			String converterBeanName = org.cattleframework.utils.auxiliary.StringUtils
				.hyphen2Camel(String.format("%s-%s-converter", beanInfo.getCategoryName(), ruleName));
			String dataSourceBeanName = org.cattleframework.utils.auxiliary.StringUtils
				.hyphen2Camel(String.format("%s-%s-data-source", beanInfo.getCategoryName(), ruleName));
			Map<String, Object> props = ruleConfigResourceProcessor.getRuleConfigResourceProps(beanFactory,
				environment, beanInfo.getCategoryName(),
				ruleName + (beanInfo.isShare() ? ""
					: "-" + SpringContext.get().getApplicationName().toLowerCase())
					+ fileExtension.toLowerCase());
			registerConverterBean(objectMapper, beanInfo.getRuleClass(),
				ruleProcessor.getConverterFactoryBeanName(fileExtension), converterBeanName);
			registerBean(ruleConfigResourceProcessor.getFactoryBeanName(), props, dataSourceBeanName,
				beanInfo.getRulePropertyDelegate(), converterBeanName);
			BaseRuleConfigResource<?> ruleConfigDataSource = beanFactory.getBean(dataSourceBeanName,
				BaseRuleConfigResource.class);
			ruleProcessor.registerProperty(ruleConfigDataSource);
		    } catch (Throwable e) {
			logger.error(String.format("规则分类'%s'的规则配置数据源'%s'构建异常:%s", beanInfo.getCategoryName(), ruleName,
				e.getMessage()), e);
			throw ExceptionWrapUtils.wrapRuntime(e);
		    }
		});
	    });
	});
    }

    private void registerConverterBean(ObjectMapper objectMapper, Class<?> ruleClass, String converterFactoryBeanName,
	    String beanName) {
	BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(converterFactoryBeanName);
	builder.addPropertyValue("objectMapper", objectMapper);
	builder.addPropertyValue("ruleClass", ruleClass);
	beanFactory.registerBeanDefinition(beanName, builder.getBeanDefinition());
    }

    private void registerBean(String factoryBeanName, Map<String, Object> props, String beanName,
	    RulePropertyDelegate<?, ?> rulePropertyDelegate, String converterBeanName) {
	BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(factoryBeanName);
	props.forEach((propName, propValue) -> {
	    builder.addPropertyValue(org.cattleframework.utils.auxiliary.StringUtils.hyphen2Camel(propName), propValue);
	});
	builder.addPropertyReference("ruleConverter", converterBeanName);
	builder.addPropertyValue("rulePropertyDelegate", rulePropertyDelegate);
	beanFactory.registerBeanDefinition(beanName, builder.getBeanDefinition());
    }
}