/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.auxiliary;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.exception.CommonException;
import org.cattleframework.exception.ExceptionWrapUtils;
import org.cattleframework.utils.reflect.ClassType;
import org.cattleframework.utils.reflect.ReflectUtils;

/**
 * 数据转换工具
 * 
 * @author orange
 *
 */
public class DataConvertUtils {

    @SuppressWarnings("unchecked")
    public static <T> T convertValue(Object value, Class<T> type) throws CommonException {
	return (T) convertValue(ReflectUtils.getClassType(type), value);
    }

    public static Object convertValue(ClassType type, Object value) throws CommonException {
	return convertValue(type, null, value);
    }

    public static Object convertValue(ClassType type, Class<?> typeClass, Object value) throws CommonException {
	try {
	    Object obj = null;
	    if (value != null) {
		ClassType valueType = ReflectUtils.getClassType(value.getClass());
		if (!(type != ClassType.String && valueType == ClassType.String
			&& StringUtils.isBlank(value.toString()))) {
		    if (type != valueType) {
			if (type == ClassType.Long) {
			    obj = convertLong(valueType, value);
			} else if (type == ClassType.BigDecimal) {
			    obj = convertBigDecimal(valueType, value);
			} else if (type == ClassType.Int) {
			    obj = convertInt(valueType, value);
			} else if (type == ClassType.Double) {
			    obj = convertDouble(valueType, value);
			} else if (type == ClassType.String) {
			    obj = convertString(valueType, value);
			} else if (type == ClassType.Array) {
			    obj = convertArray(typeClass, valueType, value);
			} else if (type == ClassType.Boolean) {
			    obj = convertBoolean(valueType, value);
			} else if (type == ClassType.Date) {
			    obj = convertDate(valueType, value);
			} else if (type == ClassType.Timestamp) {
			    obj = convertTimestamp(valueType, value);
			} else if (type == ClassType.Time) {
			    obj = convertTime(valueType, value);
			} else if (type == ClassType.SqlDate) {
			    obj = convertSqlDate(valueType, value);
			} else if (type == ClassType.Object) {
			    obj = value;
			} else if (type == ClassType.Enum) {
			    obj = convertEnum(typeClass, valueType, value);
			} else if (type == ClassType.ClassObject) {
			    obj = convertClassObject(typeClass, valueType, value);
			}
			if (obj == null) {
			    throw new CommonException("没有相应转换机制(数据类型:" + valueType.toString() + ",字段类型:"
				    + type.toString()
				    + (type == ClassType.ClassObject ? "[类:" + typeClass.getCanonicalName() + "]" : "")
				    + ")");
			}
		    } else {
			if (type == ClassType.ClassObject && typeClass != value.getClass()
				&& !typeClass.isAssignableFrom(value.getClass())) {
			    throw new CommonException("类没有相应转换机制(数据类:" + value.getClass().getCanonicalName() + ",字段类:"
				    + typeClass.getCanonicalName() + ")");
			} else if (type == ClassType.Array && ReflectUtils.getClassType(typeClass) != ReflectUtils
				.getClassType(value.getClass().getComponentType())) {
			    throw new CommonException("数组没有相应转换机制(数据类:" + value.getClass().getCanonicalName() + ",字段类:"
				    + typeClass.getCanonicalName() + "[])");
			} else if (type == ClassType.Enum
				&& !typeClass.getCanonicalName().equals(value.getClass().getCanonicalName())) {
			    throw new CommonException("枚举没有相应转换机制(数据类:" + value.getClass().getCanonicalName() + ",字段类:"
				    + typeClass.getCanonicalName() + ")");
			}
			obj = value;
		    }
		}
	    }
	    return obj;
	} catch (Throwable e) {
	    throw ExceptionWrapUtils.wrap(e);
	}
    }

    private static Object convertString(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Array) {
	    ClassType itemType = ReflectUtils.getClassType(value.getClass().getComponentType());
	    if (itemType == ClassType.Byte) {
		result = Base64.encodeBase64String((byte[]) value);
	    }
	} else {
	    result = value.toString();
	}
	return result;
    }

    private static Object convertArray(Class<?> typeClass, ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.String) {
	    ClassType itemType = ReflectUtils.getClassType(typeClass);
	    if (itemType == ClassType.Byte) {
		result = Base64.decodeBase64((String) value);
	    }
	}
	return result;
    }

    private static Object convertBoolean(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.BigDecimal) {
	    result = ((BigDecimal) value).intValue() == 1 ? true : false;
	} else if (valueType == ClassType.String) {
	    result = Boolean.parseBoolean((String) value);
	}
	return result;
    }

    private static Object convertDate(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Time) {
	    result = new java.util.Date(((java.sql.Time) value).getTime());
	} else if (valueType == ClassType.Timestamp) {
	    result = new java.util.Date(((java.sql.Timestamp) value).getTime());
	} else if (valueType == ClassType.SqlDate) {
	    result = new java.util.Date(((java.sql.Date) value).getTime());
	} else if (valueType == ClassType.Long) {
	    result = new java.util.Date((Long) value);
	}
	return result;
    }

    private static Object convertTimestamp(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Date) {
	    result = new java.sql.Timestamp(((java.util.Date) value).getTime());
	} else if (valueType == ClassType.Time) {
	    result = new java.sql.Timestamp(((java.sql.Time) value).getTime());
	} else if (valueType == ClassType.SqlDate) {
	    result = new java.sql.Timestamp(((java.sql.Date) value).getTime());
	}
	return result;
    }

    private static Object convertTime(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Date) {
	    result = new java.sql.Time(((java.util.Date) value).getTime());
	} else if (valueType == ClassType.SqlDate) {
	    result = new java.sql.Time(((java.sql.Date) value).getTime());
	} else if (valueType == ClassType.Timestamp) {
	    result = new java.sql.Time(((java.sql.Timestamp) value).getTime());
	}
	return result;
    }

    private static Object convertSqlDate(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Date) {
	    result = new java.sql.Date(((java.util.Date) value).getTime());
	} else if (valueType == ClassType.Timestamp) {
	    result = new java.sql.Date(((java.sql.Timestamp) value).getTime());
	} else if (valueType == ClassType.Time) {
	    result = new java.sql.Date(((java.sql.Time) value).getTime());
	}
	return result;
    }

    private static Object convertEnum(Class<?> typeClass, ClassType valueType, Object value) throws CommonException {
	Object result = null;
	if (valueType == ClassType.String && typeClass != null) {
	    result = ReflectUtils.setEnumValue(typeClass, (String) value);
	}
	return result;
    }

    private static Object convertClassObject(Class<?> typeClass, ClassType valueType, Object value)
	    throws URISyntaxException {
	Object result = null;
	if (valueType == ClassType.String && typeClass == URI.class) {
	    result = new URI((String) value);
	}
	return result;
    }

    private static Object convertDouble(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.BigDecimal) {
	    result = ((BigDecimal) value).doubleValue();
	} else if (valueType == ClassType.String) {
	    result = Double.valueOf(value.toString());
	}
	return result;
    }

    private static Object convertInt(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.BigDecimal) {
	    result = ((BigDecimal) value).intValue();
	} else if (valueType == ClassType.String) {
	    result = Integer.valueOf(value.toString());
	} else if (valueType == ClassType.Boolean) {
	    result = ((Boolean) value) ? 1 : 0;
	} else if (valueType == ClassType.BigInteger) {
	    result = ((BigInteger) value).intValue();
	} else if (valueType == ClassType.Long) {
	    result = ((Long) value).intValue();
	}
	return result;
    }

    private static Object convertBigDecimal(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.Int) {
	    result = new BigDecimal((Integer) value);
	} else if (valueType == ClassType.String) {
	    result = new BigDecimal(value.toString());
	} else if (valueType == ClassType.Long) {
	    result = new BigDecimal((Long) value);
	} else if (valueType == ClassType.Double) {
	    result = new BigDecimal((Double) value);
	} else if (valueType == ClassType.Boolean) {
	    result = ((Boolean) value) ? BigDecimal.ONE : BigDecimal.ZERO;
	}
	return result;
    }

    private static Object convertLong(ClassType valueType, Object value) {
	Object result = null;
	if (valueType == ClassType.BigDecimal) {
	    result = ((BigDecimal) value).longValue();
	} else if (valueType == ClassType.String) {
	    result = Long.valueOf(value.toString());
	} else if (valueType == ClassType.Int) {
	    result = ((Integer) value).longValue();
	} else if (valueType == ClassType.BigInteger) {
	    result = ((BigInteger) value).longValue();
	} else if (valueType == ClassType.Short) {
	    result = ((Short) value).longValue();
	}
	return result;
    }
}