/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils;

import java.util.Iterator;
import java.util.Map;

import org.cattleframework.aop.annotation.GuiceBindBean;
import org.cattleframework.utils.cache.CacheManager;
import org.cattleframework.utils.cache.caffeine.CaffeineCacheManager;
import org.cattleframework.utils.redis.RedisAutoConfiguration;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.context.support.MessageSourceAccessor;
import org.springframework.context.support.ResourceBundleMessageSource;

/**
 * 工具自动配置
 * 
 * @author orange
 *
 */
@AutoConfiguration
@Import(RedisAutoConfiguration.class)
@EnableConfigurationProperties(MessageSourceProperties.class)
public class UtilsAutoConfiguration {

    @Bean
    public MessageSource messageSource(MessageSourceProperties messageSourceProperties) {
	ResourceBundleMessageSource resourceBundleMessageSource = new ResourceBundleMessageSource();
	Map<String, String> baseName = messageSourceProperties.getBaseName();
	if (baseName != null && baseName.size() > 0) {
	    String[] baseNames = new String[baseName.size()];
	    Iterator<String> iterator = baseName.keySet().iterator();
	    int index = 0;
	    while (iterator.hasNext()) {
		String key = iterator.next();
		String lName = baseName.get(key) + "/" + key;
		baseNames[index] = lName;
		index++;
	    }
	    resourceBundleMessageSource.setBasenames(baseNames);
	}
	resourceBundleMessageSource.setDefaultEncoding("UTF-8");
	return resourceBundleMessageSource;
    }

    @Bean
    public MessageSourceAccessor messageSourceAccessor(MessageSource messageSource) {
	return new MessageSourceAccessor(messageSource);
    }

    @Bean
    @ConditionalOnClass(name = "com.github.benmanes.caffeine.cache.Caffeine")
    @GuiceBindBean
    public CacheManager caffeineCacheManager() {
	return new CaffeineCacheManager();
    }
}