/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.exception;

import java.util.Arrays;

import org.cattleframework.utils.reflect.ClassUtils;

/**
 * 异常定制化
 * 
 * @author orange
 *
 */
public abstract class BaseExceptionCustomizer {

    private final Class<?>[] exceptionClasses;

    /**
     * 处理异常
     * 
     * @param exception 异常
     * @return 异常处理反馈
     */
    protected abstract ExceptionProcessResponse processException(Throwable exception);

    public BaseExceptionCustomizer(Class<?>... exceptionClasses) {
	this.exceptionClasses = exceptionClasses;
    }

    public ExceptionProcessResponse process(Throwable exception) {
	Throwable throwable = getException(exception);
	if (throwable != null) {
	    return processException(throwable);
	}
	return null;
    }

    private Throwable getException(Throwable exception) {
	Class<?> clazz = ClassUtils.getUserClass(exception.getClass());
	if (Arrays.stream(exceptionClasses).anyMatch(exceptionClass -> exceptionClass.isAssignableFrom(clazz))) {
	    return exception;
	} else {
	    Throwable causeException = exception.getCause();
	    if (causeException != null) {
		return getException(causeException);
	    }
	}
	return null;
    }
}